package com.wss.common.results;

import com.google.gson.Gson;
import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.common.results.enums.LevelEnum;
import com.wss.common.results.enums.StageEnum;
import com.wss.common.results.repoResultsDTO.RepoResObjDTO;
import com.wss.common.results.repoResultsDTO.RepoResultsDTO;
import com.wss.common.results.scaResultsDTO.ScaResObjDTO;
import com.wss.common.results.scaResultsDTO.ScaResultsDTO;

import lombok.Getter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Arye.Hochman
 */
public class RepoRes {
    @Getter
    private static final int MAX_RESULTS_AMOUNT = 75;
    private static final Logger logger = LoggerFactory.getLogger(RepoRes.class);

    public static void scaResToRepoRes(LogContext logContext, ScaResultsDTO scaRes, Map<String, String> extraProps) {
        List<Map.Entry<String, List<ScaResObjDTO>>> sortedScaRes = sortScaRes(scaRes);
        RepoResultsDTO repoRes = new RepoResultsDTO(scaRes.getTotalSuccess(), scaRes.getTotalFail());

        int resCount = 0;
        for (Map.Entry<String, List<ScaResObjDTO>> path : sortedScaRes) {
            filterPsbWarnsIfScanSuccess(path.getValue());
            for (ScaResObjDTO resObj : path.getValue()) {
                if (!resObj.isSuccess()) {
                    resCount++;
                    if (resCount > MAX_RESULTS_AMOUNT) {
                        logger.info(LogUtils.formatLogMessage(logContext, "The number of SCA results is more than {}," +
                                " taking only the first {} results"), MAX_RESULTS_AMOUNT, MAX_RESULTS_AMOUNT);
                        repoRes.setLimited(MAX_RESULTS_AMOUNT);
                        break;
                    }
                    addRepoResObj(path.getKey(), resObj, repoRes);
                }
            }
            if (repoRes.getLimited() > 0) {
                break;
            }
        }

        String repoResults = new Gson().toJson(repoRes);
        extraProps.put("scaResults", repoResults);
    }

    private static List<Map.Entry<String, List<ScaResObjDTO>>> sortScaRes(ScaResultsDTO scaRes) {
        return scaRes.getResults().entrySet().stream()
                .sorted(Comparator.comparing(RepoRes::parseTimeStamp))
                .collect(Collectors.toList());
    }

    private static LocalDateTime parseTimeStamp(Map.Entry<String, List<ScaResObjDTO>> entry) {
        String time = entry.getValue().get(0).getTime();
        DateTimeFormatter formatter = new DateTimeFormatterBuilder()
                .appendPattern("yyyy-MM-dd'T'HH:mm:ss")
                .appendFraction(ChronoField.MICRO_OF_SECOND, 0, 9, true)
                .appendPattern("'Z'")
                .toFormatter();
        return LocalDateTime.parse(time, formatter);
    }

    private static void filterPsbWarnsIfScanSuccess(List<ScaResObjDTO> resObjs) {
        if (resObjs.stream()
                .filter(resObj -> resObj.getStage().equals(StageEnum.RESOLUTION.name()))
                .allMatch(resObj -> resObj.getLevel().equals(LevelEnum.INFO.name()))) {

            resObjs.stream()
                    .filter(resObj -> resObj.getStage().equals(StageEnum.PRE_STEP.name())
                            && ScaRes.supportedPMs.contains(resObj.getPm())
                            && !resObj.getExtra().containsKey("alwaysAlert"))
                    .forEach(resObj -> resObj.setSuccess(true));
        }
    }

    private static void addRepoResObj(String path, ScaResObjDTO resObj, RepoResultsDTO repoResults) {
        String pm = resObj.getPm().isEmpty() ? "general" : resObj.getPm();

        Map<String, List<Map<String, Map<String, List<RepoResObjDTO>>>>> repoRes = repoResults.getResults();
        if (!repoRes.containsKey(pm)) {
            repoRes.put(pm, new ArrayList<>());
        }

        List<Map<String, Map<String, List<RepoResObjDTO>>>> pmRes = repoRes.get(pm);
        if (pmRes.stream().noneMatch(pathPair -> pathPair.containsKey(path))) {
            pmRes.add(new HashMap<>() {{
                put(path, new HashMap<>());
            }});
        }

        Map<String, List<RepoResObjDTO>> stages = pmRes.stream()
                .filter(pathPair -> pathPair.containsKey(path)).findFirst().get().get(path);
        if (!stages.containsKey(resObj.getStage())) {
            stages.put(resObj.getStage(), new ArrayList<>());
        }

        stages.get(resObj.getStage()).add(new RepoResObjDTO(
                resObj.getLevel(), resObj.getResType(), resObj.getResMsg()));
    }
}
