import _ast
import ast
import codecs
import io
import ntpath
import os
import sys
from _ast import AST
from builtins import super
from itertools import cycle

from sources.representation.ARNodes import File, Class, Method, Identifier, Assignment, MethodInvocation, Literal, \
    Return, MemberExpression, Import, ImportAll, ARNode, DefaultAssignment, UnpackArgument
from sources.representation.ModulePathsMapper import ModulePathsMapper

EMPTY_COMMENT = "#"
PYTHON2_UNICODE_DECLARATION = "# -*- encoding: utf-8 -*-"
PY_EXTENSION = ".py"
INIT_PY = "__init__.py"
IMPORT_NODE_LINENO_KEY = "import_node_lineno"
TEMP_ID_PREFIX = "___ws___temp___"
WS_COLLECTION = "WSCollection"
WS_COLLECTION_ACCESS = "item"
WS_COLLECTION_APPEND = "append"
WS_COLLECTION_ADD = "add"
WS_IMPORT_STUB_OBJECT = "WS_IMPORT_STUB"
WS_ENTER = "__enter__"
WS_EXIT = "__exit__"


def stub_printer(x):
    pass


def parse(file_path, imports_file, printer=stub_printer):
    module_paths_mapper = ModulePathsMapper(imports_file)
    if sys.version_info[0] < 3:
        visitor = ARParserVisitor2(printer, file_path, module_paths_mapper)
    else:
        if sys.version_info[1] < 8:
            visitor = ARParserVisitor(printer, file_path, module_paths_mapper)
        else:
            visitor = ARParserVisitor38(printer, file_path, module_paths_mapper)
    with open_file_with_encoding(file_path) as f:
        content = f.read()
        ast_root = ast_parse_python_version_independent(content, file_path)
    context = ParsingContext()
    visitor.visit(ast_root, context)
    file = File(os.path.abspath(file_path), filepath_to_filename(file_path))
    file.extend_with_context(context)
    return file


def ast_parse_python_version_independent(content, file_path):
    content = remove_encoding_declaration(content)
    return ast.parse(content, file_path)


def remove_encoding_declaration(content):
    return content.replace(PYTHON2_UNICODE_DECLARATION, EMPTY_COMMENT)


def detect_bom(path, default):
    with open(path, 'rb') as f:
        raw = f.read(4)  # will read less if the file is smaller
    # BOM_UTF32_LE's start is equal to BOM_UTF16_LE so need to try the former first
    for enc, boms in \
            ('utf-8-sig', (codecs.BOM_UTF8,)), \
            ('utf-32', (codecs.BOM_UTF32_LE, codecs.BOM_UTF32_BE)), \
            ('utf-16', (codecs.BOM_UTF16_LE, codecs.BOM_UTF16_BE)):
        if any(raw.startswith(bom) for bom in boms):
            return enc
    return default


def open_file_with_encoding(file_path):
    enc = detect_bom(file_path, 'utf-8')
    return io.open(file_path, "rt", encoding=enc)


def filepath_to_filename(path):
    head, tail = ntpath.split(path)
    return tail or ntpath.basename(head)


def iter_fields(node):
    """
    Yield a tuple of ``(fieldname, value)`` for each field in ``node._fields``
    that is present on *node*.
    """
    for field in node._fields:
        try:
            yield field, getattr(node, field)
        except AttributeError:
            pass


def is_literals(*args):
    return all(type(arg) is Literal for arg in args)


class ASTContextVisitor(object):
    def visit(self, node, context, **kwargs):
        """Visit a node."""
        method = 'visit_' + node.__class__.__name__
        visitor = getattr(self, method, self.generic_visit)
        return visitor(node, context, **kwargs)

    def generic_visit(self, node, context, **kwargs):
        """Called if no explicit visitor function exists for a node."""
        for field, value in iter_fields(node):
            if isinstance(value, list):
                for item in value:
                    if isinstance(item, AST):
                        self.visit(item, context, **kwargs)
            elif isinstance(value, AST):
                self.visit(value, context, **kwargs)


class ParsingContext:
    def __init__(self):
        self.results = []
        self.cfg = []
        self.declarations = []

    def merge(self, other):
        self.cfg.extend(other.cfg)
        self.declarations.extend(other.declarations)

    def get_all(self):
        res = []
        res.extend(self.results)
        res.extend(self.cfg)
        res.extend(self.declarations)
        return res


# noinspection PyPep8Naming,PyMethodMayBeStatic
class ARParserVisitor(ASTContextVisitor):
    def __init__(self, printer, filepath, module_paths_mapper):
        self.printer = printer
        self.filepath = filepath
        self.module_paths_mapper = module_paths_mapper

    def visit(self, node, context, **kwargs):
        super(ARParserVisitor, self).visit(node, context, **kwargs)
        for arnode in context.get_all():
            if isinstance(arnode, ARNode) and not hasattr(arnode, "position") and hasattr(node, "lineno"):
                arnode.add_position(node.lineno, node.col_offset)

    def generic_visit(self, node, context, **kwargs):
        self.printer("Unhandled node of type: " + node.__class__.__name__)
        if hasattr(node, "lineno"):
            self.printer("@position: line=" + str(node.lineno) + " col=" + str(node.col_offset))
        if hasattr(node, "_fields"):
            self.printer("Fields: " + str(node._fields))
            super(ARParserVisitor, self).generic_visit(node, context, **kwargs)

    def visit_Module(self, node, context, **kwargs):
        # fields: body
        self.visitList(node.body, context, **kwargs)
        self.remove_hanging_nodes_from_context(context)

    def visit_ClassDef(self, node, context, **kwargs):
        # fields: name, bases, keywords, body, decorator_list
        classContext = ParsingContext()
        self.visitList(node.body, classContext, **kwargs)
        self.remove_hanging_nodes_from_context(classContext)

        if not any(x for x in classContext.declarations if x.name == "__init__"):
            init_method = Method("__init__", ["self"])
            init_method.add_position(node.lineno, node.col_offset)
            classContext.declarations.insert(0, init_method)

        baseContext = ParsingContext()
        self.visitList(node.bases, baseContext, **kwargs)
        context.merge(baseContext)

        class_declaration = Class(node.name, baseContext.results)
        class_declaration.extend_with_context(classContext)

        context.declarations.append(class_declaration)

    def visitList(self, list_, context, **kwargs):
        for item in list_:
            self.visit(item, context, **kwargs)

    def remove_hanging_nodes_from_context(self, context):
        self.remove_hanging_nodes_from_list(context.cfg)
        self.remove_hanging_nodes_from_list(context.results)

    def remove_hanging_nodes_from_list(self, cfg):
        hanging = [node for node in cfg if self.is_hanging(node)]
        for node in hanging:
            cfg.remove(node)

    def is_hanging(self, node):
        return type(node) in [Literal, Identifier, MemberExpression]

    def visit_FunctionDef(self, node, context, **kwargs):
        # fields: args, body, decorator_list, name
        methodContext = ParsingContext()
        self.visitList(node.body, methodContext, **kwargs)
        self.remove_hanging_nodes_from_context(methodContext)
        argsContext = ParsingContext()
        self.visit(node.args, argsContext, **kwargs)

        varargs_name = self.get_arg_name(node.args.vararg)
        kwargs_name = self.get_arg_name(node.args.kwarg)
        method = Method(node.name, argsContext.results, self.is_static_method(node), varargs_name, kwargs_name)
        method.extend_with_context(methodContext)
        method.cfg.extend(argsContext.cfg)

        context.declarations.append(method)

        decorator_parsing_ctx = ParsingContext()
        self.visitList(node.decorator_list, decorator_parsing_ctx, **kwargs)
        context.merge(decorator_parsing_ctx)

        for res in decorator_parsing_ctx.results:
            method_id = Identifier(node.name)
            dec_assignment = Assignment(method_id, MethodInvocation(res, [method_id]))
            context.cfg.append(dec_assignment)

    def get_arg_name_with_args(self, node):
        return node.arg

    def get_arg_name(self, arg):
        return arg.arg if arg is not None else None

    def visit_arguments(self, node, context, **kwargs):
        # node fields: args, defaults, kw_defaults, kwarg, kwonlyargs, vararg
        arg_names = [self.get_arg_name_with_args(x) for x in node.args]
        context.results.extend(arg_names)

        self.add_default_parameters_to_cfg(node, context, kwargs)

    def add_default_parameters_to_cfg(self, node, context, kwargs):
        def_params = self.get_def_params(node)
        for def_param, def_ast_value in zip(def_params, node.defaults):
            def_value_context = ParsingContext()
            self.visit(def_ast_value, def_value_context, **kwargs)
            context.merge(def_value_context)
            for def_value_result in def_value_context.results:
                ass = DefaultAssignment(Identifier(def_param), def_value_result)
                context.cfg.append(ass)

    def get_def_params(self, arguments_node):
        arg_names = [self.get_arg_name_with_args(x) for x in arguments_node.args]
        num_defaults = len(arguments_node.defaults)
        return arg_names[-num_defaults:]

    def visit_Assign(self, node, context, **kwargs):
        # node fields: targets, value
        for target in node.targets:
            self.parse_single_assignment(target, node.value, context, **kwargs)

    def visit_AnnAssign(self, node, context, **kwargs):
        # fields: target, annotation, value, simple
        self.parse_single_assignment(node.target, node.value, context, **kwargs)

    def parse_single_assignment(self, target, value, context, **kwargs):
        lhsContext = ParsingContext()
        rhsContext = ParsingContext()
        self.visit(target, lhsContext, **kwargs)
        self.visit(value, rhsContext, **kwargs)
        not_literal = False
        for rhsResult in rhsContext.results:
            if not isinstance(rhsResult, Literal):
                not_literal = True
        if not_literal:
            context.merge(lhsContext)
            context.merge(rhsContext)
            cfg_extension = []
            for lhsResult in lhsContext.results:
                for rhsResult in rhsContext.results:
                    if type(lhsResult) is WSCollectionAddIntermediateResult:
                        if not lhsResult.is_literal_only_args(rhsResult):
                            cfg_extension.append(lhsResult.get_add_invocation(rhsResult))
                    elif type(lhsResult) is WSCollectionLHSStoreIntermediateResult:
                        rhs_temp_id, rhs_temp_ass = self.create_temp_assignment(rhsResult, value.lineno,
                                                                                value.col_offset)
                        context.cfg.append(rhs_temp_ass)
                        element_access_assigments = lhsResult.get_elements_assignments_to_collection_access(rhs_temp_id)
                        context.cfg.extend(element_access_assigments)
                    else:
                        assignment = Assignment(lhsResult, rhsResult)
                        cfg_extension.append(assignment)
            context.cfg.extend(cfg_extension)

    def visit_Call(self, node, context, **kwargs):
        # node fields: func, args, keywords
        funcContext = ParsingContext()
        self.visit(node.func, funcContext, **kwargs)
        context.merge(funcContext)
        argsContext = ParsingContext()
        self.visitList(node.args, argsContext, **kwargs)
        context.merge(argsContext)
        args = argsContext.results
        named = self.merge_call_args(args, context, kwargs, node)
        for target in funcContext.results:
            method_invocation_node = MethodInvocation(target, args, named)
            context.results.append(method_invocation_node)

    def merge_call_args(self, args, context, kwargs, node):
        if len(node.keywords) > 0 or hasattr(node, "starargs") or hasattr(node, "kwargs"):
            named = [keyword.arg for keyword in node.keywords if keyword.arg is not None]
            keyword_values = [keyword.value for keyword in node.keywords if keyword.arg is not None]
            kw_values_context = ParsingContext()
            self.visitList(keyword_values, kw_values_context, **kwargs)
            context.merge(kw_values_context)
            args.extend(kw_values_context.results)
            self.merge_keywords(args, context, keyword_values, kwargs, node)
        else:
            named = []
        return named

    def merge_keywords(self, args, context, keyword_values, kwargs, node):
        starred_kwargs = [keyword for keyword in node.keywords if keyword.arg is None]
        starred_context = ParsingContext()
        for starred_kwarg in starred_kwargs:
            self.parseStarredArgument(starred_context, starred_kwarg.value, **kwargs)
        context.merge(starred_context)
        args.extend(starred_context.results)

    def visit_Attribute(self, node, context, **kwargs):
        # node fields: value, attr, ctx
        object_context = ParsingContext()
        self.visit(node.value, object_context, **kwargs)
        context.merge(object_context)
        for result in object_context.results:
            member_exp = MemberExpression(result, node.attr)
            context.results.append(member_exp)

    def visit_UnaryOp(self, node, context, **kwargs):
        # fields: op, operand
        unaryOpContext = ParsingContext()
        self.visit(node.operand, unaryOpContext, **kwargs)
        context.merge(unaryOpContext)
        results = unaryOpContext.results
        if type(node.op) is ast.USub and len(results) == 1 and type(results[0]) is Literal:
            literal = results[0]
            if literal.literalType == Literal.LITERAL_INT or literal.literalType == Literal.LITERAL_FLOAT:
                literal.value = -literal.value
        context.results.extend(unaryOpContext.results)

    def visit_Name(self, node, context, **kwargs):
        id = Identifier(node.id)
        context.results.append(id)

    def visit_Str(self, node, context, **kwargs):
        literal = Literal(Literal.LITERAL_STRING, node.s)
        context.results.append(literal)

    def visit_Num(self, node, context, **kwargs):
        value = node.n
        if type(value) is int:
            literal = Literal(Literal.LITERAL_INT, value)
        elif type(value) is float or type(value) is complex:
            literal = Literal(Literal.LITERAL_FLOAT, value)
        else:
            raise Exception("Unsupported literal num type [type=" + str((type(value)) + "]"))
        context.results.append(literal)

    def visit_NameConstant(self, node, context, **kwargs):
        # fields: value
        if node.value is None:
            literal = Literal(Literal.LITERAL_NONE, None)
        else:
            literal = Literal(Literal.LITERAL_BOOL, node.value)
        context.results.append(literal)

    def visit_Return(self, node, context, **kwargs):
        # fields: value
        returnContext = ParsingContext()
        self.visit(node.value, returnContext, **kwargs)
        context.merge(returnContext)
        for result in returnContext.results:
            return_node = Return(result)
            context.cfg.append(return_node)

    def visit_List(self, node, context, **kwargs):
        # fields: elts, ctx
        elements_context = ParsingContext()
        self.visitList(node.elts, elements_context, **kwargs)
        context.merge(elements_context)

        if type(node.ctx) is _ast.Store:
            ir = WSCollectionLHSStoreIntermediateResult(elements_context.results)
            context.results.append(ir)
        else:
            temp_id, list_assignment = self.create_temp_assignment(self.get_wsc_creation_mi(),
                                                                   node.lineno,
                                                                   node.col_offset)
            context.cfg.append(list_assignment)

            for result in [e for e in elements_context.results if not is_literals(e)]:
                appendCall = self.create_wsc_append(temp_id, result)
                context.cfg.append(appendCall)

            context.results.append(temp_id)

    def create_wsc_append(self, collection_id, item):
        return MethodInvocation(MemberExpression(collection_id, WS_COLLECTION_APPEND), [item])

    def get_wsc_creation_mi(self):
        return MethodInvocation(Identifier(WS_COLLECTION), [])

    def visit_Subscript(self, node, context, **kwargs):
        # fields: value, slice, ctx
        value_context = ParsingContext()
        self.visit(node.value, value_context, **kwargs)
        context.merge(value_context)
        sliceContext = ParsingContext()
        self.visit(node.slice, sliceContext, **kwargs)
        context.merge(sliceContext)

        for collection_node in value_context.results:
            if type(node.slice) is _ast.Index:
                if type(node.ctx) is _ast.Load or type(node.ctx) is _ast.Del:
                    collection_access_call = self.create_collection_access(collection_node)
                    context.results.append(collection_access_call)
                    context.cfg.extend(sliceContext.results)
                elif type(node.ctx) is _ast.Store:  # "Store"
                    for result in sliceContext.results:
                        add_node = MethodInvocation(MemberExpression(collection_node, WS_COLLECTION_ADD), [result])
                        append_command = WSCollectionAddIntermediateResult(add_node)
                        context.results.append(append_command)
                else:
                    raise Exception("Subscript index context type ins't supported. [node: " + str(node.__dict__) + "]")
            if type(node.slice) is _ast.Slice:
                if type(collection_node) is not Literal:
                    context.results.append(collection_node)
                context.cfg.extend(sliceContext.results)

    def visit_For(self, node, context, **kwargs):
        # Fields: ('target', 'iter', 'body', 'orelse')
        iterContext = ParsingContext()
        self.visit(node.iter, iterContext, **kwargs)
        context.merge(iterContext)

        targetContext = ParsingContext()
        self.visit(node.target, targetContext, **kwargs)
        context.merge(targetContext)

        iter_temp_id = self.generate_temp_id(node.iter.lineno, node.iter.col_offset)
        for target in targetContext.results:
            for iter in iterContext.results:
                iter_assign = Assignment(iter_temp_id, iter)
                context.cfg.append(iter_assign)
            iter_item_invocation = self.create_collection_access(iter_temp_id)
            if isinstance(target, WSCollectionLHSStoreIntermediateResult):
                assignments = target.get_elements_assignments_to_collection_access(iter_temp_id)
                context.results.extend(assignments)
            else:
                item_assign = Assignment(target, iter_item_invocation)
                context.results.append(item_assign)

        self.visitList(node.body, context, **kwargs)
        self.visitList(node.orelse, context, **kwargs)

    def visit_With(self, node, context, **kwargs):
        # Fields: ('items', 'body')

        withItemsContext = ParsingContext()
        self.visitList(node.items, withItemsContext, **kwargs)
        context.merge(withItemsContext)
        context.results.extend(withItemsContext.results)
        self.visitList(node.body, context, **kwargs)

    def visit_withitem(self, node, context, **kwargs):
        # Fields: ('context_expr', 'optional_vars')
        # context_expr holds the withitem's managed resource allocation
        # optional vars holds the variable that is assigned to the allocated resource
        exprContext = ParsingContext()
        self.visit(node.context_expr, exprContext, **kwargs)
        context.merge(exprContext)
        i = 0
        for result in exprContext.results:
            temp_id, temp_expr_assign = self.create_temp_assignment(result, node.context_expr.lineno,
                                                                    node.context_expr.col_offset, i)
            i = i + 1
            context.results.append(temp_expr_assign)
            enter_call = MethodInvocation(MemberExpression(Identifier(temp_id.name), WS_ENTER), [])

            if node.optional_vars is not None:
                varContext = ParsingContext()
                self.visit(node.optional_vars, varContext, **kwargs)
                context.merge(varContext)
                self.assert_single_result(varContext, node.optional_vars)
                var = varContext.results[0]

                item_assign = Assignment(var, enter_call)
                context.results.append(item_assign)
            else:
                context.results.append(enter_call)

            exit_call = MethodInvocation(MemberExpression(Identifier(temp_id.name), WS_EXIT), [])

            context.results.append(exit_call)

    def assert_single_result(self, context, node):
        if len(context.results) != 1:
            raise Exception(str(type(node)) + " without one result are not supported [" + str(node.__dict__) + "]")

    def visit_ExceptHandler(self, node, context, **kwargs):
        # Fields: ('type', 'name', 'body')
        if node.name is not None:
            exceptionTypesContext = ParsingContext()
            self.visit(node.type, exceptionTypesContext, **kwargs)
            context.merge(exceptionTypesContext)

            self.assert_single_result(exceptionTypesContext, node.type)
            exception_type = exceptionTypesContext.results[0]
            if isinstance(node.type, _ast.Tuple):
                exception_type = self.create_collection_access(exception_type)

            exception_assign = Assignment(Identifier(self.get_identifier_name(node)),
                                          MethodInvocation(exception_type, []))
            context.results.append(exception_assign)
        self.visitList(node.body, context, **kwargs)

    def get_identifier_name(self, node):
        return node.name

    def create_collection_access(self, collection_id):
        return MethodInvocation(MemberExpression(collection_id, WS_COLLECTION_ACCESS), [])

    def visit_AugAssign(self, node, context, **kwargs):
        # Fields: ('target', 'op', 'value')
        self.parse_single_assignment(node.target, node.value, context, **kwargs)

    def visit_Lambda(self, node, context, **kwargs):
        # fields: args, body
        lambdaContext = ParsingContext()
        self.visit(node.body, lambdaContext, **kwargs)
        lambda_body = [Return(result) for result in lambdaContext.results]
        argsContext = ParsingContext()
        self.visit(node.args, argsContext, **kwargs)
        temp_lambda_id = self.generate_temp_id(node.lineno, node.col_offset)
        vararg = self.get_arg_name(node.args.vararg)
        kwarg = self.get_arg_name(node.args.kwarg)
        lambda_method_dec = Method(temp_lambda_id.name, argsContext.results, self.is_static_method(node), vararg, kwarg)
        lambda_method_dec.declarations.extend(lambdaContext.declarations)
        lambda_method_dec.cfg.extend(lambdaContext.cfg)
        lambda_method_dec.cfg.extend(lambda_body)
        context.merge(argsContext)
        context.declarations.append(lambda_method_dec)
        context.results.append(temp_lambda_id)

    def visit_Import(self, node, context, **kwargs):
        # fields: names
        kwargs[IMPORT_NODE_LINENO_KEY] = node.lineno
        self.visitList(node.names, context, **kwargs)

    def visit_alias(self, node, context, **kwargs):
        # fields: name, asname
        ar_import = self.create_import(node.name)
        if node.asname:
            import_assign = Assignment(Identifier(node.asname), ar_import)
            context.results.append(import_assign)
        elif "." not in node.name:
            context.results.append(Assignment(Identifier(node.name), ar_import))
        else:
            self.get_dotted_imports_assignments(node.name, context, **kwargs)

    def create_import(self, name, relative_level=0):
        imported_filepath = self.to_filepath(name, relative_level)
        return Import(name, imported_filepath)

    def to_filepath(self, module_name, relative_level):
        imported_filepath = self.module_paths_mapper.to_filepath(self.filepath, module_name, relative_level)
        return imported_filepath

    def get_dotted_imports_assignments(self, module_name, context, level=0, **kwargs):

        parts = module_name.split(".")
        sub_imports = [parts[0]]
        for part in parts[1:]:
            sub_import = sub_imports[-1] + "." + part
            sub_imports.append(sub_import)

        sub_import_resolved = False
        for sub_import in reversed(sub_imports):
            lhs = self.string_to_id_or_member_exp(sub_import)
            imported_filepath = self.to_filepath(sub_import, level)
            if imported_filepath and not self.is_same_file(imported_filepath):
                sub_import_resolved = True
                assert imported_filepath.endswith(PY_EXTENSION)
                rhs = Import(sub_import, imported_filepath)
                context.cfg.append(Assignment(lhs, rhs))
            elif sub_import_resolved:
                rhs = self.get_import_stub_invocation()
                context.cfg.append(Assignment(lhs, rhs))

    def is_same_file(self, imported_filepath):
        return os.path.realpath(imported_filepath) == os.path.realpath(self.filepath)

    def get_import_stub_invocation(self):
        return MethodInvocation(Identifier(WS_IMPORT_STUB_OBJECT), [])

    def visit_ImportFrom(self, node, context, **kwargs):
        # fields: module, names, level
        module = node.module if node.module else ""
        self.get_dotted_imports_assignments(module, context, node.level, **kwargs)
        for alias in node.names:
            # from-import aliases can only be simple identifiers (without dots)
            if alias.name == "*":
                module = module + "*"
                context.results.append(ImportAll(module, self.to_filepath(module, node.level)))
            else:
                import_assignment = self.getImportAssignmentFromAlias(alias, node.module, node.level)
                context.results.append(import_assignment)

    def getImportAssignmentFromAlias(self, alias, imported_module, import_level, ):
        if alias.asname is None:
            assign_lhs = Identifier(alias.name)
        else:
            assign_lhs = Identifier(alias.asname)
        module_name = imported_module if imported_module else ""
        imported_file = module_name + alias.name
        filepath = self.to_filepath(imported_file, import_level)

        if not os.path.exists(filepath):
            filepath = self.to_filepath(imported_module, import_level)

        assign_rhs = Import(imported_file, filepath)
        if self.is_submodule_import(filepath, module_name, alias.name):
            assign_rhs = MemberExpression(assign_rhs, alias.name)
        import_assignment = Assignment(assign_lhs, assign_rhs)
        return import_assignment

    def is_submodule_import(self, filepath, module_name, name):
        is_init_file = filepath.endswith(INIT_PY)
        dir_name = os.path.dirname(filepath)
        if is_init_file and dir_name.endswith(name):
            module_parts = module_name.split(".")
            cur_dirname = os.path.dirname(dir_name)
            is_module_name = True
            while module_parts:
                is_module_name = is_module_name and cur_dirname.endswith(module_parts[-1])
                module_parts = module_parts[:-1]
                cur_dirname = os.path.dirname(cur_dirname)
            if is_module_name:
                return False

        module_suffix = module_name.replace(".", os.sep) + os.sep + name + PY_EXTENSION
        return not filepath.endswith(module_suffix)

    def get_x_parent_folders_relative_path(self, x):
        return "../".join(["" for _ in range(x)])

    def get_import_name_and_type(self, import_node):
        if isinstance(import_node, Import):
            imported_name = imported_type = import_node.name
        elif isinstance(import_node, Assignment) and isinstance(import_node.rhs, Import):
            imported_name = import_node.lhs.name
            imported_type = import_node.rhs.name
        else:
            raise Exception("Couldn't resolve import name and type from import node: " + str(import_node.__dict__))
        return imported_name, imported_type

    def generate_temp_id(self, lineno, colno, additional=None):
        return Identifier(self.generate_temp_str(colno, lineno, additional))

    def generate_temp_str(self, colno, lineno, additional=None):
        return TEMP_ID_PREFIX + str(lineno) + "_" + str(colno) + (("_" + str(additional)) if additional else "")

    def visit_Set(self, node, context, **kwargs):
        # fields: elts
        eltContext = ParsingContext()
        self.visitList(node.elts, eltContext, **kwargs)
        context.merge(eltContext)

        set_creation = MethodInvocation(Identifier("set"), [])
        temp_id, set_assign = self.create_temp_assignment(set_creation, node.lineno, node.col_offset)
        context.cfg.append(set_assign)
        for set_item in eltContext.results:
            prerequisite = MethodInvocation(MemberExpression(Identifier(temp_id.name), "add"), [set_item])
            context.cfg.append(prerequisite)
        context.results.append(temp_id)

    def visit_DictComp(self, node, context, **kwargs):
        # fields: key, value, generators
        generators_context = ParsingContext()
        self.visitList(node.generators, generators_context, **kwargs)
        context.merge(generators_context)
        context.cfg.extend(generators_context.results)

        assignment, temp_id = self.create_new_wsc(node.lineno, node.col_offset)
        context.cfg.append(assignment)
        context.results.append(temp_id)

        key_context = ParsingContext()
        self.visit(node.key, key_context, **kwargs)
        context.merge(key_context)

        value_context = ParsingContext()
        self.visit(node.value, value_context, **kwargs)
        context.merge(value_context)

        for key in key_context.results:
            for value in value_context.results:
                if not is_literals(key, value):
                    wsc_dictionary_add_invocation = self.create_dict_add_invocation(temp_id, key, value)
                    context.cfg.append(wsc_dictionary_add_invocation)

    def visit_Dict(self, node, context, **kwargs):
        # fields: keys, values
        assignment, temp_id = self.create_new_wsc(node.lineno, node.col_offset)
        context.cfg.append(assignment)
        context.results.append(temp_id)

        self.add_dict_elements_to_cfg(node, temp_id, context, kwargs)

    def create_new_wsc(self, lineno, col_offset):
        dict_creation = self.get_wsc_creation_mi()
        temp_id, assignment = self.create_temp_assignment(dict_creation, lineno, col_offset)
        return assignment, temp_id

    def add_dict_elements_to_cfg(self, dict_node, dict_id, context, kwargs):
        keys_context = ParsingContext()
        values_context = ParsingContext()
        self.visitList(dict_node.keys, keys_context, **kwargs)
        self.visitList(dict_node.values, values_context, **kwargs)
        context.merge(keys_context)
        context.merge(values_context)
        keys = keys_context.results
        values = values_context.results
        k_v_pairs = zip(cycle(keys), values) if len(values) > len(keys) else zip(keys, cycle(values))
        for k, v in k_v_pairs:
            if not is_literals(k, v):
                add_invocation = self.create_dict_add_invocation(dict_id, k, v)
                context.cfg.append(add_invocation)

    def create_dict_add_invocation(self, dict_id, key, value):
        add_invocation = MethodInvocation(MemberExpression(dict_id, WS_COLLECTION_ADD), [key, value])
        return add_invocation

    def create_temp_assignment(self, assigned_node, lineno, col_offset, additional=None):
        temp_id = self.generate_temp_id(lineno, col_offset, additional)
        assignment = Assignment(temp_id, assigned_node)
        return temp_id, assignment

    def visit_IfExp(self, node, context, **kwargs):
        # fields: test, body, orelse
        test_context = ParsingContext()
        self.visit(node.test, test_context, **kwargs)
        context.merge(test_context)
        context.cfg.extend(test_context.results)

        self.visit(node.body, context, **kwargs)
        self.visit(node.orelse, context, **kwargs)

    def visit_Tuple(self, node, context, **kwargs):
        # fields: elts, ctx
        self.visit_List(node, context, **kwargs)

    def visit_Compare(self, node, context, **kwargs):
        # fields: left, ops, comparators
        new_context = ParsingContext()
        self.generic_visit(node, new_context, **kwargs)
        context.merge(new_context)
        context.cfg.extend(new_context.results)
        context.results.append(Literal(Literal.LITERAL_BOOL, True))

    def visit_GeneratorExp(self, node, context, **kwargs):
        # fields: elt, generators
        self.visit_ListComp(node, context, **kwargs)

    def visit_ListComp(self, node, context, **kwargs):
        # fields: elt, generators
        generators_context = ParsingContext()
        self.visitList(node.generators, generators_context, **kwargs)
        context.merge(generators_context)
        context.cfg.extend(generators_context.results)

        new_wsc = self.get_wsc_creation_mi()
        temp_id, assignment = self.create_temp_assignment(new_wsc, node.lineno, node.col_offset)
        context.cfg.append(assignment)

        elt_context = ParsingContext()
        self.visit(node.elt, elt_context, **kwargs)
        context.merge(elt_context)
        for elt in [e for e in elt_context.results if not is_literals(e)]:
            elt_append = self.create_wsc_append(temp_id, elt)
            context.cfg.append(elt_append)
        context.results.append(temp_id)

    def visit_comprehension(self, node, context, **kwargs):
        # fields: target, iter, ifs, is_async
        self.visitList(node.ifs, context, **kwargs)

        iter_context = ParsingContext()
        self.visit(node.iter, iter_context, **kwargs)
        context.merge(iter_context)

        target_context = ParsingContext()
        self.visit(node.target, target_context, **kwargs)
        context.merge(target_context)

        for target in target_context.results:
            for collection in iter_context.results:
                collection_item = self.create_collection_access(collection)
                if isinstance(target, WSCollectionLHSStoreIntermediateResult):
                    context.results.extend(target.get_elements_assignments_to_collection_access(collection_item))
                else:
                    context.results.append(Assignment(target, collection_item))

    def visit_Starred(self, node, context, **kwargs):
        # fields: value, ctx
        self.parseStarredArgument(context, node.value, **kwargs)

    def parseStarredArgument(self, context, starred_expr, **kwargs):
        value_context = ParsingContext()
        self.visit(starred_expr, value_context, **kwargs)
        context.merge(value_context)
        for res in value_context.results:
            starred_arg = UnpackArgument(res)
            context.results.append(starred_arg)

    def string_to_id_or_member_exp(self, name):
        if "." not in name:
            return Identifier(name)
        else:
            parts = name.split(".")
            member = parts[-1]
            object_string = ".".join(parts[:-1])
            return MemberExpression(self.string_to_id_or_member_exp(object_string), member)

    def is_static_method(self, node):
        if hasattr(node, 'decorator_list'):
            return len(list(filter(lambda x: (hasattr(x, 'id') and x.id == 'staticmethod'), node.decorator_list))) > 0
        return False


class ARParserVisitor38(ARParserVisitor):

    def visit_Constant(self, node, context, **kwargs):
        value = node.n
        if type(value) is int:
            literal = Literal(Literal.LITERAL_INT, value)
        elif type(value) is float or type(value) is complex:
            literal = Literal(Literal.LITERAL_FLOAT, value)
        elif type(value) is str or type(value) is complex:
            literal = Literal(Literal.LITERAL_STRING, value)
        elif type(value) is bool or type(value) is complex:
            literal = Literal(Literal.LITERAL_BOOL, value)
        elif type(value) is type(None) or type(value) is complex:
            literal = Literal(Literal.LITERAL_NONE, value)
        elif type(value) is bytes:
            literal = Literal(Literal.LITERAL_BYTES, str(value))
        else:
            raise Exception("Unsupported literal num type [type=" + str((type(value)) + "]"))
        context.results.append(literal)


class ARParserVisitor2(ARParserVisitor):
    def merge_keywords(self, args, context, keyword_values, kwargs, node):
        if node.kwargs:
            kwargs_context = ParsingContext()
            self.parseStarredArgument(kwargs_context, node.kwargs, **kwargs)
            context.merge(kwargs_context)
            args.extend(kwargs_context.results)
        if node.starargs:
            starred_context = ParsingContext()
            self.parseStarredArgument(starred_context, node.starargs, **kwargs)
            context.merge(starred_context)
            args.extend(starred_context.results)

    def get_arg_name(self, arg):
        return arg if arg is not None and isinstance(arg, basestring) else None

    def get_identifier_name(self, node):
        return node.name.id

    def get_arg_name_with_args(self, node):
        if isinstance(node, _ast.Name):
            return node.id
        else:
            return node

    def visit_Name(self, node, context, **kwargs):
        if node.id == 'None':
            context.results.append(Literal(Literal.LITERAL_NONE, None))
        else:
            id = Identifier(node.id)
            context.results.append(id)

    def visit_With(self, node, context, **kwargs):
        withItemsContext = ParsingContext()
        self.visit_withitem(node, context, **kwargs)
        context.merge(withItemsContext)
        context.results.extend(withItemsContext.results)
        self.visitList(node.body, context, **kwargs)


class WSCollectionAddIntermediateResult(object):
    """
    Helper class for parsing subscript store operations through assignments, e.g.,
    x[0] = value
    will translate to
    x.add(value)
    """

    def __init__(self, add_command_without_value):
        self.add_command_without_value = add_command_without_value

    def get_add_invocation(self, value):
        self.add_command_without_value.arguments.append(value)
        return self.add_command_without_value

    def is_literal_only_args(self, value):
        first_arg = self.add_command_without_value.arguments[0]
        return is_literals(first_arg, value)


class WSCollectionLHSStoreIntermediateResult(object):
    """
    Helper class for parsing store operations of collections through assignments, e.g.
    [x,y] = array
    will translate to
    x = array.item
    y = array.item
    """

    def __init__(self, assigned_elements):
        self.assigned_elements = assigned_elements

    def get_elements_assignments_to_collection_access(self, collection_id, level=0):
        res = []
        for element in self.assigned_elements:
            if type(element) is WSCollectionLHSStoreIntermediateResult:
                res.extend(element.get_elements_assignments_to_collection_access(collection_id, level + 1))
            elif type(element) is WSCollectionAddIntermediateResult:
                res.append(element.get_add_invocation(self._get_access_invocation_(collection_id, level)))
            elif type(element) is UnpackArgument:
                unpackingRes = WSCollectionLHSStoreIntermediateResult(
                    [element.exp]).get_elements_assignments_to_collection_access(collection_id, level - 1)
                res.extend(unpackingRes)
            else:
                res.append(Assignment(element, self._get_access_invocation_(collection_id, level)))
        return res

    @staticmethod
    def _get_access_invocation_(col_id, level):
        if level < 0:
            return col_id
        access = MethodInvocation(MemberExpression(col_id, WS_COLLECTION_ACCESS), [])
        for i in range(level):
            access = MethodInvocation(MemberExpression(access, WS_COLLECTION_ACCESS), [])
        return access
