const fs = require('fs');
const path = require('path');
const babel = require("@babel/core");

function transpileFiles(inputPath, files, outputPath){

    try{
        console.log(`Transpiling ${files.length} files`)
        const start = new Date().getTime();

        const errorFiles = {}
        let counter = 0;
        files.forEach((file) => {
            transpileFile(inputPath, file, outputPath, errorFiles, babelOptions);
            counter++;
            if (counter === 1 || counter % 1000 === 0 || counter === files.length) {
                console.log(`Transpiled ${counter} out of ${files.length} files`)
            }
        });

        const numOfFailedFiles = Object.keys(errorFiles).length;
        console.log(`numOfFailedFiles: ${numOfFailedFiles}`)

        const end = new Date().getTime();
        console.log(`Runtime: ${end - start} ms`)
        console.log(`exitCode: ${numOfFailedFiles}`);
        return numOfFailedFiles;
    } catch (error) {
        console.log(`Transpilation failed on folder ${inputPath}: ${error.message}`)
        console.log(`exitCode: -1`);
        return -1;
    }
}


function transpileFile(inputPath, relativeFile, outputPath, errorFiles, options){

    try {

        let code, map;
        let result = {code, map};

        const sourceFile = path.join(inputPath, relativeFile);
        const sourceCode = fs.readFileSync(sourceFile, 'utf8');
        let modifiedOptions = createOptions(sourceFile, options);

        try {
            result = babel.transformSync(sourceCode, modifiedOptions);
        }
        catch (error) {
            modifiedOptions = createOptions(sourceFile, options, typescriptSyntaxPlugin);
            result = babel.transformSync(sourceCode, modifiedOptions);
        }

        const outputFilePath = path.join(outputPath, relativeFile);
        const outputMapPath = outputFilePath + ".map";

        fs.mkdirSync( path.dirname(outputFilePath), { recursive: true })

        fs.writeFileSync(outputFilePath, result.code, 'utf8');
        fs.writeFileSync(outputMapPath, JSON.stringify(result.map), 'utf8');

        if (result.metadata && result.metadata.customInfo && result.metadata.customInfo.nestEntries){
            const entryFile = outputFilePath + ".nestjs.entries";
            fs.writeFileSync(entryFile, JSON.stringify(result.metadata.customInfo.nestEntries), 'utf8');
        }

        if (result.metadata && result.metadata.customInfo && result.metadata.customInfo.decTypes){
            const entryFile = outputFilePath + ".dec.types";
            let unifiedTypes = unifyTypesAndImportInformation(result.metadata.customInfo.decTypes, result.metadata.customInfo.importDeclarationMap)
            fs.writeFileSync(entryFile, JSON.stringify(unifiedTypes), 'utf8');
        }

    } catch (error) {

        console.log(`Error transpiling file: ${relativeFile}` /*${error.message}`*/);
        errorFiles[relativeFile] = error.message;

        //Write the file as failed:
        const outputFilePath = path.join(outputPath, relativeFile+".error");
        fs.mkdirSync( path.dirname(outputFilePath), { recursive: true })
        fs.writeFileSync(outputFilePath, error.message, 'utf8');
    }

}

function unifyTypesAndImportInformation(decTypes, importDeclarationMap){

    // Create a Set to store unique objects
    const uniqueObjects = [];

    // Iterate through the array and add unique objects to the Set
    decTypes.forEach(decType => {

        if (importDeclarationMap) {
            let importFile = importDeclarationMap.get(decType.type);
            if (importFile && !importFile.startsWith("./") && !importFile.startsWith("../") && !importFile.startsWith(".\\") && !importFile.startsWith("..\\")) {
                let tempType = decType.type;
                decType.type = importFile + "." + tempType;
            }
        }

        let isUnique = true;

        // Check if the object is already in the Set based on property comparison
        for (const uniqueObj of uniqueObjects) {
            if (areDecTypeEqual(uniqueObj, decType)) {
                isUnique = false;
                break;
            }
        }

        if (isUnique) {
            uniqueObjects.push(decType);
        }
    });

    return uniqueObjects;

}

function areDecTypeEqual(decType1, decType2) {
    const keysA = Object.keys(decType1);
    const keysB = Object.keys(decType2);

    // Check if the number of keys is the same
    if (keysA.length !== keysB.length) {
        return false;
    }

    // Check if all keys and their values are equal
    for (const key of keysA) {
        if (decType1[key] !== decType2[key]) {
            return false;
        }
    }

    return true;
}


function getAllFilesInFolder(inputPath) {
    let fileList = [];

    function traverseDirectory(currentPath) {
        const files = fs.readdirSync(currentPath);

        files.forEach((file) => {

            try {
                const filePath = path.join(currentPath, file);
                if (fs.statSync(filePath).isDirectory()) {
                    // If it's a directory, recursively traverse it
                    traverseDirectory(filePath);
                } else {
                    // If it's a file, add it to the list
                    fileList.push( path.sep + path.relative(inputPath, filePath) );
                }
            }
            catch (error){
                //Do nothing. Ignore this file
            }

        });
    }

    //Check if inputPath exist
    if ( !fs.existsSync(inputPath) ){
        throw new Error('The given path does not exist')
    }

    let stats = fs.statSync(inputPath)
    if (stats.isDirectory()) {
        traverseDirectory(inputPath);
    }
    else{
     throw new Error('The given path is not a folder')
    }

    //Return the filtered list
    return fileList;
}

function createOptions(sourceFile, options, additionalPlugin){

    let modifiedOptions = JSON.parse(JSON.stringify(options));
    const ext = path.extname(sourceFile);
    if (ext === ".ts" || ext === ".tsx" || ext === ".mts" || ext === ".cts") {
        modifiedOptions.presets.unshift(typescriptPreset);
        modifiedOptions.plugins.splice(2, 0, typescriptPlugin);
    }
    if (additionalPlugin != null){
        modifiedOptions.plugins.splice(2, 0, additionalPlugin);
    }
    modifiedOptions.filename = sourceFile;

    return modifiedOptions;

}

const babelOptions = {
    presets: [
        "@babel/preset-react",
        "@babel/preset-flow"
    ],
    plugins: [
        ["./plugins/saveInformation"],
        ["./plugins/removeDecorators"],
        ["@babel/plugin-syntax-import-attributes",{"deprecatedAssertSyntax":true}],
        "@babel/plugin-transform-class-properties",
        "@babel/plugin-transform-private-methods",
        ["@babel/plugin-proposal-decorators",{"legacy":true}],
        "@babel/plugin-transform-modules-commonjs",
        "@babel/plugin-transform-export-namespace-from",
        "@babel/plugin-proposal-export-default-from"
    ],
    sourceType: 'unambiguous',
    sourceMaps: true
};

const typescriptPreset = [
    "@babel/preset-typescript",{"allowDeclareFields":true}
]

const typescriptPlugin = [
    "@babel/plugin-transform-typescript",{"allowDeclareFields":true}
]

const typescriptSyntaxPlugin = [
    "@babel/plugin-syntax-typescript"
]


module.exports = { transpileFiles, transpileFile, getAllFilesInFolder };
