package com.wss.scanner.registry.utils.registryHandlers.nuget;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.scanner.registry.models.HostRule;
import com.wss.scanner.registry.utils.filesParsers.XmlParser;
import com.wss.scanner.registry.utils.registryHandlers.PrivateRegistryFileHandler;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.net.URISyntaxException;
import java.util.*;

import static com.wss.scanner.registry.utils.Constants.*;
import static com.wss.scanner.registry.utils.PrivateRegistryUtils.getParentElementFromDoc;
import static com.wss.scanner.registry.utils.PrivateRegistryUtils.writeDocToXmlFile;

public class NugetPrivateRegistryHandler extends PrivateRegistryFileHandler {
    private final static Logger logger = LoggerFactory.getLogger(NugetPrivateRegistryHandler.class);
    private static final String CLEAR_TEXT_PASSWORD = "ClearTextPassword";
    private static final String USERNAME = "Username";
    private static final String MINUS_SOURCE = "-Source";
    private static final String MINUS_VERBOSITY = "-Verbosity";
    private static final String QUIET = "quiet";
    private static final String MINUS_INTERACTIVE = "-NonInteractive";

    @Override
    public void createRegistryObject(LogContext logContext, String registryObject, List<HostRule> hostRules) {
        try {
            Document baseNugetConfigDoc = getBaseNugetConfigDoc(logContext);
            enrichDocObjectAndWriteToFile(logContext, registryObject, hostRules, baseNugetConfigDoc);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - createRegistryObject - error while creating the {} file "), registryObject);
        }
    }

    @Override
    public void editRegistryObject(LogContext logContext, String registryFilePath, List<HostRule> hostRules) {
        try {
            Document nugetConfigDoc = new XmlParser().parseXmlFile(logContext, registryFilePath, true);
            clearNugetConfigElement(logContext, registryFilePath, nugetConfigDoc, PACKAGE_SOURCES);
            clearNugetConfigElement(logContext, registryFilePath, nugetConfigDoc, PACKAGE_SOURCE_CREDENTIALS);
            enrichDocObjectAndWriteToFile(logContext, registryFilePath, hostRules, nugetConfigDoc);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - editRegistryObject - error while editing the registry file {} "), registryFilePath);
        }
    }

    @Override
    public boolean isProjectLevelRegistryFile() {
        return true;
    }

    @Override
    public boolean isSystemLevelRegistryFile() {
        return false;
    }

    @Override
    public boolean isEnvVariableRegistryCredentials() {
        return false;
    }

    @Override
    public void prepareIncludesAndExcludes(ArrayList<String> includes, ArrayList<String> excludes) {
        String[] nugetIncludedManifests = new String[]{
                //DirectoryScanner is set to case insensitive, so we can just put the lower here.
                TWO_STARS_SLASH + NUGET_CONFIG_LOWER,
                TWO_STARS_SLASH + STAR + DOT_NUSPEC,
                TWO_STARS_SLASH + STAR + DOT_CSPROJ,
                TWO_STARS_SLASH + STAR + DOT_SLN,
                TWO_STARS_SLASH + PACKAGES_CONFIG
        };

        includes.addAll(Arrays.asList(nugetIncludedManifests));
    }

    @Override
    public List<String> getManifestTypes(String[] localManifestsFiles) {
        return Arrays.asList(localManifestsFiles);
    }

    @Override
    public List<String> getGlobalRegistryObject(boolean isFile) {
        return null;
    }

    @Override
    public List<String> getRegistryFileType(String manifestFile) {
        if (manifestFile.endsWith(NUGET_CONFIG_LOWER)) {
            return Arrays.asList(NUGET_CONFIG_LOWER);
        } else {
            return Arrays.asList(NUGET_CONFIG_UPPER);
        }
    }


    /* private methods */
    /**
     * This method loads the default NuGet.Config file which is saved in the resources folder.
     * @param logContext - the log context
     * @return Document object representing the NuGet.Config file
     */
    private Document getBaseNugetConfigDoc(LogContext logContext) {
        logger.info(LogUtils.formatLogMessage(logContext, "Loading the base NuGet.Config from the resources " +
                "folder"));
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        java.net.URL resource = classLoader.getResource(NUGET_CONFIG_UPPER);
        Document doc = null;
        try {
            java.net.URI uri = resource.toURI();
            doc =  new XmlParser().parseXmlFile(logContext, String.valueOf(uri), false);
        } catch (URISyntaxException e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - getBaseNugetConfigDoc - error while loading base NuGet.Config "));
        }
        return doc;
    }

    /**
     * In this method we enrich the doc object with the information from the host rules list and save it to a xml file
     * @param logContext - the log context
     * @param filePath   - the file path we want to write the document object to
     * @param hostRules  - the host rules lise
     * @param doc        - the document object we want to edit
     */
    private void enrichDocObjectAndWriteToFile(LogContext logContext, String filePath, List<HostRule> hostRules,
                                               Document doc) {
        if (doc != null) {
            addElementsToXmlDoc(logContext, doc, hostRules);
            writeDocToXmlFile(logContext, doc, filePath);
        }
    }

    /**
     * In this method we add the needed Document elements to the original doc
     * @param logContext - the log context
     * @param document   - the document holding the xml lines
     * @param hostRules  - the host rules
     */
    private void addElementsToXmlDoc(LogContext logContext, Document document, List<HostRule> hostRules) {
        logger.info(LogUtils.formatLogMessage(logContext, "Adding Elements to the NuGet.Config Document"));
        Element configuration = (Element) document.getElementsByTagName(CONFIGURATION).item(0);
        Element packageSourcesElement = getParentElementFromDoc(document, configuration, PACKAGE_SOURCES);
        Element packageSourceCredentialsElement = getParentElementFromDoc(document, configuration, PACKAGE_SOURCE_CREDENTIALS);

        int ruleIndex = 1;
        for (HostRule rule : hostRules) {
            String hostUrl = rule.getMatchHost();
            String ruleName = WS_RULE_NAME + ruleIndex;

            // add elements to packageSources
            addElementToParent(document, packageSourcesElement, ruleName, hostUrl);

            String userName = rule.getUserName();
            String credentials = StringUtils.isNotBlank(rule.getToken()) ? rule.getToken() : rule.getPassword();
            if (StringUtils.isNotBlank(userName) && StringUtils.isNotBlank(credentials)) {
                // add elements to packageSourceCredentials
                Element userPasswordWrappingElement = document.createElement(ruleName);
                addElementToParent(document, userPasswordWrappingElement, USERNAME, userName);
                addElementToParent(document, userPasswordWrappingElement, CLEAR_TEXT_PASSWORD, credentials);
                packageSourceCredentialsElement.appendChild(userPasswordWrappingElement);
            } else if (StringUtils.isBlank(userName)){
                logger.info(LogUtils.formatLogMessage(logContext, "no user name was provided, skipping!"));
            } else {
                logger.info(LogUtils.formatLogMessage(logContext, "no token or password was provided, skipping!"));
            }

            ruleIndex++;
        }
    }

    /**
     * In this method we append the new "add" element to its parent element with the given key/ value as attributes
     * @param document      - the full document
     * @param parentElement - parent element to append the newly created element to
     * @param key           - the key
     * @param value         - the value
     */
    private void addElementToParent(Document document, Element parentElement, String key, String value) {
        Element element = document.createElement(ADD);
        element.setAttribute(KEY, key);
        element.setAttribute(VALUE, value);
        parentElement.appendChild(element);
    }

    /**
     * In this method we read the lines of the registry file and strip all the not configured env variables found in the
     * lines
     * @param logContext       - the log context
     * @param registryFilePath - the registry file path
     */
    private void clearNugetConfigElement(LogContext logContext, String registryFilePath, Document doc, String elementName) {
        NodeList element = doc.getElementsByTagName(elementName);
        for (int i = 0; i < element.getLength(); i++) {
            Node node = element.item(i);
            node.getParentNode().removeChild(node);
        }
        doc.createElement(elementName);
    }
}
