package com.wss.scanner.registry.utils.registryHandlers.gradle;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.scanner.registry.utils.OsUtils;
import com.wss.scanner.registry.utils.filesParsers.TextFileParser;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.wss.scanner.registry.utils.Constants.*;
import static com.wss.scanner.registry.utils.Constants.EMPTY_STRING;

public class GradleConfigurationsUtils {
    private final static Logger logger = LoggerFactory.getLogger(GradleConfigurationsUtils.class);
    private final static String DOT_GRADLE = ".gradle";
    private final static String EXT = "ext";
    private final static String GRADLE_USER_HOME = "GRADLE_USER_HOME";
    private final static String ORG_GRADLE_PROJECT = "ORG_GRADLE_PROJECT_";
    private final static String SYSTEM_PROP = "systemProp.";
    private final static String APPLY_FROM = "apply from:";

    /**
     * In this method we extract properties that gradle can use from multiple places. in the following order:
     * 1. from the project level "gradle.properties" (if found)
     * 2. from the home level "gradle.properties" (if found)
     * 3. from the env variables defined in the system level (start with ORG_GRADLE_PROJECT_)
     * 4. from the system prop configured as "systemProp.<prop>"
     * 5. another properties file pointed with "apply from:" from the registry file
     * note, if the same key is repeated, then the later take precedence
     *
     * @param logContext        - the log context
     * @param projectDirPath    - the project dir path
     * @param registryFileLines - the registry file lines
     * @return a map with all the properties found in the above files
     */
    public Map<String, String> buildGradleProperties(LogContext logContext, String projectDirPath,
                                                     List<String> registryFileLines) {
        Map<String, String> allProperties = new HashMap<>();
        Map<String, String> envVariables = System.getenv();

        try {
            extractPropertiesFromProjectLevelFile(logContext, projectDirPath, allProperties);
            extractPropertiesFromHomeDirFile(logContext, allProperties, envVariables);
            extractPropertiesFromEnvVariables(logContext, allProperties, envVariables);
            extractPropertiesFromSysProps(logContext, allProperties);
            extractPropertiesFromRegistryFileAndExtBlocks(logContext, projectDirPath, registryFileLines, allProperties);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - buildGradleProperties - error while collecting gradle properties "));
        }

        // print all properties keys
        for (String propKey : allProperties.keySet()) {
            logger.info(LogUtils.formatLogMessage(logContext, "found property key: {}"), propKey);
        }
        return allProperties;
    }

    /* private methods */

    /**
     * In this method we locate parse the project level "properties.gradle" file
     *
     * @param logContext     - the log context
     * @param projectDirPath - the project directory path
     * @param allProperties  - all properties map
     */
    private void extractPropertiesFromProjectLevelFile(LogContext logContext, String projectDirPath,
                                                       Map<String, String> allProperties) {
        try {
            TextFileParser textFileParser = new TextFileParser();
            String localGradlePropertiesPath = Paths.get(projectDirPath, GRADLE_PROPERTIES).toString();
            if (new File(localGradlePropertiesPath).exists()) {
                Map<String, String> localProperties = textFileParser.parseTxtFileLineByLineWithDelimiter(logContext,
                        localGradlePropertiesPath, EQUALS);
                allProperties.putAll(localProperties);
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - extractPropertiesFromProjectLevelFile - error while extracting Properties From Project Level properties "));
        }
    }

    /**
     * In this method we locate and parse the global "properties.gradle" file (home user file)
     *
     * @param logContext    - the log context
     * @param allProperties - all properties map
     * @param envVariables  - the env variables map
     */
    private void extractPropertiesFromHomeDirFile(LogContext logContext, Map<String, String> allProperties,
                                                  Map<String, String> envVariables) {
        try {
            String gradleHomeDir = "";
            TextFileParser textFileParser = new TextFileParser();
            File gradleDefaultHomeDir = new File(OsUtils.getHomeDirectory(), DOT_GRADLE);

            // first we check if the user configured the gradle home directory as a system prop and look for the
            // "gradle.properties" at this location
            if (allProperties.containsKey("systemProp.gradle.user.home")) {
                gradleHomeDir = allProperties.get("systemProp.gradle.user.home");
            } else if (envVariables.containsKey(GRADLE_USER_HOME)) {
                // we look for the home directory as a variable configured by: "GRADLE_USER_HOME" in the env variables
                gradleHomeDir = envVariables.get(GRADLE_USER_HOME);
            } else if (gradleDefaultHomeDir.exists()) {
                // last option is to look for the "properties.gradle" in the default home directory of the system
                gradleHomeDir = gradleDefaultHomeDir.toString();
            }

            File globalGradlePropertiesFile = new File(gradleHomeDir, GRADLE_PROPERTIES);
            if (globalGradlePropertiesFile.exists()) {
                Map<String, String> globalProperties = textFileParser.parseTxtFileLineByLineWithDelimiter(logContext,
                        globalGradlePropertiesFile.toString(), EQUALS);
                allProperties.putAll(globalProperties);
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - extractPropertiesFromHomeDirFile - error while extracting Properties From home directory " +
                    "properties file "));
        }
    }

    /**
     * In this method we search the registry lines for lines with "apply from:", which indicates a pointer to a specific
     * properties file to extract the properties from.
     * also, we search for any property configured in as "ext.<prop>" or in the "ext {" block
     *
     * @param logContext        - the log context
     * @param projectPath       - the project path dir
     * @param registryFileLines - the registry file lines
     * @param allProperties     - already found properties map
     */
    private void extractPropertiesFromRegistryFileAndExtBlocks(LogContext logContext, String projectPath,
                                                               List<String> registryFileLines,
                                                               Map<String, String> allProperties) {
        boolean isExtBlock = false;
        String extDot = EXT + DOT;
        String EXT_BRACKET = EXT;

        try {
            for (String line : registryFileLines) {
                String lineTrim = line.trim();
                if (StringUtils.isBlank(lineTrim)) {
                    continue;
                }

                if (lineTrim.contains(extDot)) {
                    String variable = lineTrim.substring(extDot.length());
                    String[] keyAndValue = variable.split(EQUALS);
                    allProperties.put(keyAndValue[0].trim(), keyAndValue[1].trim());
                } else if (lineTrim.startsWith(EXT_BRACKET) && lineTrim.endsWith(OPEN_CURLY_BRACKETS)) {
                    isExtBlock = true;
                } else if (isExtBlock) {
                    if (lineTrim.endsWith(CLOSE_CURLY_BRACKETS)) {
                        isExtBlock = false;
                    } else {
                        String[] keyAndValue = lineTrim.split(EQUALS);
                        allProperties.put(keyAndValue[0].trim(), keyAndValue[1].trim());
                    }
                } else if (isDefinedVariable(lineTrim)) {
                    String keyValue = lineTrim.replace("def", "").trim();
                    String[] KV = keyValue.split("=");
                    allProperties.put(KV[0].trim(), KV[1].trim());

                } else if (lineTrim.contains(APPLY_FROM)) {
                    extractPropertiesFromApplyFromFile(logContext, projectPath, allProperties, lineTrim);
                }
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - extractPropertiesFromRegistryFileAndExtBlocks - error while extracting Properties From " +
                    "Registry File And Ext Blocks "));
        }
    }

    /**
     * In this method we located and parse the properties file pointed with the "apply from:" line
     *
     * @param logContext     - the log context
     * @param projectDirPath - the project directory path
     * @param allProperties  - the properties already parsed
     * @param line           - the apply from line
     */
    private void extractPropertiesFromApplyFromFile(LogContext logContext, String projectDirPath,
                                                    Map<String, String> allProperties, String line) {
        // example: "apply from: 'repositories.gradle', to: scriptHandler" or "apply from: 'modules.gradle'"
        String fileName = "";
        int commaIndex = line.indexOf(COMMA);
        if (commaIndex == -1) {
            fileName = line.substring(APPLY_FROM.length()).trim();
        } else {
            fileName = line.substring(APPLY_FROM.length(), commaIndex).trim();
        }

        // remove ' or "
        String fileNameClean = fileName.replaceAll(QUOTATION_MARK, EMPTY_STRING).replaceAll(APOSTROPHE, EMPTY_STRING);
        File file = new File(fileNameClean);
        File fileWithProjectFolder = new File(projectDirPath, fileNameClean);
        Map<String, String> applyFromProperties = new HashMap<>();
        TextFileParser textFileParser = new TextFileParser();
        if (file.exists()) {
            // look for the file as is (in case it's a full path)
            applyFromProperties = textFileParser.parseTxtFileLineByLineWithDelimiter(logContext,
                    file.toString(), EQUALS);
        } else if (fileWithProjectFolder.exists()) {
            // look for the file name inside the project folder
            applyFromProperties = textFileParser.parseTxtFileLineByLineWithDelimiter(logContext,
                    fileWithProjectFolder.toString(), EQUALS);
        }
        allProperties.putAll(applyFromProperties);
    }

    /**
     * In this method we extract all the env variables properties that start with "ORG_GRADLE_PROJECT_"
     *
     * @param allProperties - all properties map
     * @param envVariables  - the system env variables map
     */
    private void extractPropertiesFromEnvVariables(LogContext logContext, Map<String, String> allProperties,
                                                   Map<String, String> envVariables) {
        try {
            for (Map.Entry<String, String> envVariable : envVariables.entrySet()) {
                String envVariableKey = envVariable.getKey();
                if (envVariableKey.startsWith(ORG_GRADLE_PROJECT)) {
                    String key = envVariableKey.substring(ORG_GRADLE_PROJECT.length());
                    String envVariableValues = envVariable.getValue();
                    allProperties.put(key.trim(), envVariableValues.trim());
                }
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - extractPropertiesFromEnvVariables - error while extracting Properties From " +
                    "environment variables "));
        }
    }

    /**
     * In this method we look for properties configured as "systemProp.<prop>" in the properties map. This properties
     * take precedence of properties with the same key
     *
     * @param allProperties - all properties map
     */
    private void extractPropertiesFromSysProps(LogContext logContext, Map<String, String> allProperties) {
        try {
            Map<String, String> sysVariables = new HashMap<>();
            for (Map.Entry<String, String> prop : allProperties.entrySet()) {
                String key = prop.getKey();
                if (key.startsWith(SYSTEM_PROP)) {
                    sysVariables.put(key.substring(SYSTEM_PROP.length()), prop.getValue());
                }
            }
            allProperties.putAll(sysVariables);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - extractPropertiesFromSysProps - error while extracting Properties From " +
                    "systemProp. variables "));
        }
    }

    /**
     * this method receives a bom file line and checks if it's a defined variable
     *
     * @param line - the line to verify
     * @return true if it is a line that defines a variable
     */
    private boolean isDefinedVariable(String line) {
        return line.matches("def\\s+.*=.*");
    }
}
