package com.wss.scanner.registry.utils.registryHandlers.go;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.scanner.registry.models.HostRule;
import com.wss.scanner.registry.utils.OsUtils;
import com.wss.scanner.registry.utils.PrivateRegistryUtils;
import com.wss.scanner.registry.utils.filesParsers.TextFileParser;
import com.wss.scanner.registry.utils.registryHandlers.PrivateRegistryFileHandler;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;

import static com.wss.scanner.registry.utils.Constants.*;

public class GoPrivateRegistryHandler extends PrivateRegistryFileHandler {
    /* --- Private Static Fields --- */
    private final static Logger logger = LoggerFactory.getLogger(GoPrivateRegistryHandler.class);
    private final static String GO_OFFICIAL_PROXY = "https://proxy.golang.org";
    private final static String DIRECT = "direct";
    private final static String NETRC = "NETRC";
    private final static String NETRC_FILE = OsUtils.isWindows() ? "_netrc" : ".netrc";
    private final static String MACHINE = "machine";
    private final static String LOGIN = "login";

    /* --- Constructors --- */
    public GoPrivateRegistryHandler() {
    }

    /* --- Public Methods --- */
    @Override
    public void createRegistryObject(LogContext logContext, String registryObject, List<HostRule> hostRules) {
        if (registryObject.endsWith(NETRC_FILE)) {
            handleNetrcFile(logContext, registryObject, hostRules, false);
        } else {
            handleEnvVariables(logContext, registryObject, hostRules, false);
        }
    }

    @Override
    public void editRegistryObject(LogContext logContext, String registryObject, List<HostRule> hostRules) {
        if (registryObject.endsWith(NETRC_FILE)) {
            handleNetrcFile(logContext, registryObject, hostRules, true);
        } else {
            handleEnvVariables(logContext, registryObject, hostRules, true);
        }
    }

    @Override
    public boolean isProjectLevelRegistryFile() {
        return false;
    }

    @Override
    public boolean isSystemLevelRegistryFile() {
        return true;
    }

    @Override
    public boolean isEnvVariableRegistryCredentials() {
        return true;
    }

    @Override
    public void prepareIncludesAndExcludes(ArrayList<String> includes, ArrayList<String> excludes) {

    }

    @Override
    public List<String> getManifestTypes(String[] localManifestsFiles) {
        return null;
    }

    @Override
    public List<String> getGlobalRegistryObject(boolean isFile) {
        if (isFile) {
            String netrcFileLocation;
            if (System.getProperties().contains(NETRC)) {
                netrcFileLocation = System.getProperty(NETRC);
            } else {
                String homeDirectory = OsUtils.getHomeDirectory();
                netrcFileLocation = homeDirectory + FileSystemSeparator + NETRC_FILE;
            }
            return Arrays.asList(netrcFileLocation);
        } else {
            return Arrays.asList(GOPROXY, GONOSUMDB);
        }
    }

    @Override
    public List<String> getRegistryFileType(String manifestFile) {
        return Arrays.asList(NETRC_FILE);
    }

    /**
     * This is a wrapper method for the createEditNetrcFile
     *
     * @param logContext       - the log context
     * @param registryFilePath - the registry file path
     * @param hostRules        - the host rules list
     * @param isFileExist      - whether the .netrc file exist or not
     */
    private void handleNetrcFile(LogContext logContext, String registryFilePath, List<HostRule> hostRules,
                                 boolean isFileExist) {
        if (isFileExist) {
            TextFileParser textFileParser = new TextFileParser();
            List<String> lines = textFileParser.parseTxtFileLineByLine(logContext, registryFilePath);
            createEditNetrcFile(logContext, registryFilePath, hostRules, lines);
        } else {
            createEditNetrcFile(logContext, registryFilePath, hostRules, new LinkedList<>());
        }
    }

    /**
     * In this method we create or edit the .netrc file content
     *
     * @param logContext       - the log context
     * @param registryFilePath - the registry file path
     * @param hostRules        - the host rules list
     * @param existingLines    - the list of the existing lines in the .netrc file
     */
    private void createEditNetrcFile(LogContext logContext, String registryFilePath, List<HostRule> hostRules,
                                     List<String> existingLines) {
        try {
            List<String> lines = prepareTextLines(logContext, hostRules, SPACE);
            Path path = Paths.get(registryFilePath);
            existingLines.addAll(lines);

            PrivateRegistryUtils.writeLinesToFile(logContext, path, existingLines);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - createNetrcFile - error while creating the netrc file "));
        }
    }

    /**
     * In this method we prepare the lines to be written to the .netrc file
     *
     * @param logContext - the log context
     * @param hostRules  - the host rules list
     * @param delimiter  - the delimiter between the words in each line
     * @return list of lines to write to the .netrc file
     */
    private List<String> prepareTextLines(LogContext logContext, List<HostRule> hostRules, String delimiter) {
        logger.info(LogUtils.formatLogMessage(logContext, "preparing the lines for the .netrc file"));
        List<String> lines = new LinkedList<>();

        for (HostRule hostRule : hostRules) {
            String host = PrivateRegistryUtils.getHostName(hostRule.getMatchHost());

            String pass;
            String login;
            if (StringUtils.isNotBlank(hostRule.getUserName())) {
                login = hostRule.getUserName();
                pass = StringUtils.isNotBlank(hostRule.getToken()) ? hostRule.getToken() : hostRule.getPassword();
            } else {
                login = hostRule.getToken();
                pass = "x-oauth-basic";
            }

            lines.add(MACHINE + delimiter + host + delimiter + LOGIN + delimiter + login + delimiter + PASSWORD +
                    delimiter + pass);
        }
        return lines;
    }

    /**
     * This method gets the expected environment variable value and sets it as a system property
     *
     * @param logContext          - the log context
     * @param registryEnvVariable - the env variable we want to set
     * @param hostRules           - the host rules list
     * @param isEnvVariableExist  - basically it says if we're in edit or create mode
     */
    private void handleEnvVariables(LogContext logContext, String registryEnvVariable, List<HostRule> hostRules,
                                    boolean isEnvVariableExist) {
        try {
            String envVariableValue = buildEnvVariableValue(logContext, registryEnvVariable, hostRules, isEnvVariableExist);
            if (envVariableValue == null) return;

            System.setProperty(SCANNER_CUSTOM_PROP_ + registryEnvVariable, envVariableValue);
        } catch (Exception e) {
            String action = isEnvVariableExist ? "edit" : "create";

            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - handleEnvVariables - error while {} the env variable {} "), action, registryEnvVariable);
        }
    }

    /**
     * This method is a wrapper of the methods that build the env variable value
     *
     * @param logContext          - the log context
     * @param registryEnvVariable - the env variable key
     * @param hostRules           - the host rules
     * @return the requested env variable value
     */
    private String buildEnvVariableValue(LogContext logContext, String registryEnvVariable, List<HostRule> hostRules,
                                         boolean envVariableExist) {
        String envVariableValue;
        if (registryEnvVariable.equals(GOPROXY)) {
            envVariableValue = buildGoProxy(logContext, hostRules, envVariableExist);
        } else if (registryEnvVariable.equals(GONOSUMDB)) {
            envVariableValue = buildGoNoSumDb(logContext, hostRules, envVariableExist);
        } else {
            return null;
        }
        return envVariableValue;
    }

    /**
     * In this method we build the GOPROXY env variable value. we build it by combining each host rule url with the
     * username and password/token, for example: "user1:pass1@host1|user2:pass2@host2|https://proxy.golang.org|direct"
     * the | (pipe) is for falling to the next proxy if the current one failed to access the dependency
     * the last arguments: "https://proxy.golang.org and direct" are the default value of GOPROXY
     * if the GOPROXY env variable was already set, we append the existing value to the new values we created
     *
     * @param logContext       - the log context
     * @param hostRules        - the host rules list
     * @param envVariableExist - edit or create mode
     * @return the full of GOPROXY value to be set as env variable
     */
    private String buildGoProxy(LogContext logContext, List<HostRule> hostRules, boolean envVariableExist) {
        logger.info(LogUtils.formatLogMessage(logContext, "Building the GOPROXY env variable"));
        StringBuilder fullGoProxy = new StringBuilder();
        for (HostRule hostRule : hostRules) {
            if (StringUtils.isNotBlank(fullGoProxy)) {
                fullGoProxy.append(PIPE);
            }
            String userName = StringUtils.isNotBlank(hostRule.getUserName()) ? hostRule.getUserName() : "";
            String tokenOrPassword;
            if (StringUtils.isNotBlank(hostRule.getToken())) {
                tokenOrPassword = hostRule.getToken();
            } else if (StringUtils.isNotBlank(hostRule.getPassword())) {
                tokenOrPassword = hostRule.getPassword();
            } else {
                tokenOrPassword = "";
                logger.debug(LogUtils.formatLogMessage(logContext, "No password or token were provided"));
            }

            String matchHost = hostRule.getMatchHost();
            String common = userName + COLON + tokenOrPassword + AT +
                    matchHost.replace(HTTP_PATTERN, EMPTY_STRING).replace(HTTPS_PATTERN, EMPTY_STRING);

            String hostWithCredentials = matchHost.startsWith(HTTP_PATTERN) ? HTTP_PATTERN + common : HTTPS_PATTERN + common;

            fullGoProxy.append(hostWithCredentials);
        }

        if (envVariableExist) {
            fullGoProxy.append(PIPE).append(System.getProperty(GOPROXY).replace(QUOTATION_MARK, EMPTY_STRING).
                    replace(APOSTROPHE, EMPTY_STRING));
        } else {
            // run `go env <KEY>` to get the default value from the .env file
            String defaultValue = getEnvVariableValueFromDotEnvFile(logContext, GOPROXY);

            if (StringUtils.isNotBlank(defaultValue)) {
                fullGoProxy.append(PIPE).append(defaultValue);
            } else {
                fullGoProxy.append(PIPE).append(GO_OFFICIAL_PROXY).append(PIPE).append(DIRECT);
            }
        }

        return fullGoProxy.toString().trim();
    }

    /**
     * In this method we build the value of the GONOSUMDB env variable which will be set from each one of the host rules
     * taking only the host (+port) and the first part of the full path and adding (*) after, for example: host1.com/user/*
     * This env variable is a comma-separated list of module suffixes that checksum database queries should be
     * disabled for. Wildcards are supported.
     *
     * @param logContext       - the log context
     * @param hostRules        - the host rules list
     * @param envVariableExist - edit or create mode
     * @return the full of GONOSUMDB value to be set as env variable
     */
    private String buildGoNoSumDb(LogContext logContext, List<HostRule> hostRules, boolean envVariableExist) {
        logger.info(LogUtils.formatLogMessage(logContext, "Building the GONOSUMDB env variable"));
        StringBuilder fullGoNoSumDb = new StringBuilder();
        for (HostRule hostRule : hostRules) {
            if (StringUtils.isNotBlank(fullGoNoSumDb)) {
                fullGoNoSumDb.append(COMMA);
            }
            String matchHost = hostRule.getMatchHost().replace(HTTP_PATTERN, EMPTY_STRING).
                    replace(HTTPS_PATTERN, EMPTY_STRING);
            int firstSlashInd = matchHost.contains(FORWARD_SLASH) ? matchHost.indexOf(FORWARD_SLASH) : matchHost.length();
            int slashInd = matchHost.indexOf(FORWARD_SLASH, firstSlashInd + 1);
            int endInd = Integer.max(firstSlashInd, slashInd);
            String subMatchHost = matchHost.substring(0, endInd);
            fullGoNoSumDb.append(subMatchHost);
            if (subMatchHost.endsWith(FORWARD_SLASH)) {
                fullGoNoSumDb.append(STAR);
            } else {
                fullGoNoSumDb.append(FORWARD_SLASH).append(STAR);
            }
        }

        if (envVariableExist) {
            fullGoNoSumDb.append(COMMA).append(System.getProperty(GONOSUMDB).replace(QUOTATION_MARK, EMPTY_STRING).
                    replace(APOSTROPHE, EMPTY_STRING));
        } else {
            // run `go env <KEY>` to get the default value from the .env file
            String defaultValue = getEnvVariableValueFromDotEnvFile(logContext, GONOSUMDB);

            if (StringUtils.isNotBlank(defaultValue)) {
                fullGoNoSumDb.append(PIPE).append(defaultValue);
            }
        }

        return fullGoNoSumDb.toString().trim();
    }

    /**
     * In this method we run the command `go env <envVariableKey>` in order to get the default value of the current env
     * variable as was set by the user (or the main default value of GO if not touched).
     *
     * @param logContext     - the log context
     * @param envVariableKey - the env variable we're looking for its value in .env file
     * @return the value of the env variable from the .env file
     */
    private String getEnvVariableValueFromDotEnvFile(LogContext logContext, String envVariableKey) {
        logger.info(LogUtils.formatLogMessage(logContext, "getting the env variable {} value from the .env file"),
                envVariableKey);
        List<String> command = new LinkedList<>(Arrays.asList(GO, ENV, envVariableKey));
        String envVariableVal = PrivateRegistryUtils.runCommandAndReadOutput(logContext, command);

        if (envVariableKey.equals(GOPROXY)) {
            return envVariableVal.replace(COMMA, PIPE);
        }
        return envVariableVal;
    }
}
