package com.wss.scanner.registry.utils;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.scanner.registry.utils.filesParsers.XmlParser;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.yaml.snakeyaml.Yaml;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.wss.scanner.registry.utils.Constants.*;
import static com.wss.scanner.registry.utils.Constants.HTTPS_PATTERN;

public class PrivateRegistryUtils {
    private final static Logger logger = LoggerFactory.getLogger(PrivateRegistryUtils.class);
    private final static Pattern ENV_VAR_PATTERN = Pattern.compile("\\$\\{([A-Za-z0-9_.-]+)(?::([^\\}]*))?\\}");
    private final static Pattern ENV_VAR_PATTERN_NO_BRACKETS = Pattern.compile("\\$([A-Za-z0-9_.-]+)(?::([^\\}]*))?");
    private final static Pattern ENV_VAR_PATTERN_WINDOWS = Pattern.compile("%(.*?)%");

    /**
     * In this method, we locate all the env variables in the registry file and convert them to the corresponding system
     * variables (if found). if not found, we remove the line.
     *
     * @param logContext       - the log context
     * @param lines            - the registry file parsed lines
     * @param registryFilePath - the registry file path
     */
    public void stripRegistryFileLines(LogContext logContext, List<String> lines, String registryFilePath) {
        Set<String> linesToRemove = new HashSet<>();

        logger.info(LogUtils.formatLogMessage(logContext, "converting environment variables in the registry" +
                "file: {}"), registryFilePath);
        Map<String, String> systemEnvVariables = System.getenv();
        for (String line : lines) {
            boolean envVariablesExist = isEnvVariablesExist(logContext, line, systemEnvVariables);

            if (!envVariablesExist) {
                linesToRemove.add(line);
            }
        }

        lines.removeAll(linesToRemove);
    }

    /**
     * In this method we locate the env variables (if any) in the text string, and check if it's configured as a
     * system variable, if the corresponding system variable is not configured, then we return false
     *
     * @param logContext         - the log context
     * @param text               - the text that we want to check if it contains env variables
     * @param systemVariablesMap - the map containing all the system variables
     * @return true if the no env variables found in the text line, or the env variables are configured as a system
     * variable, else return false
     */
    public boolean isEnvVariablesExist(LogContext logContext, String text, Map<String, ?> systemVariablesMap) {
        try {
            // check if the env variables is ${ENV}
            if (!checkEnvVariableExistInText(text, systemVariablesMap, ENV_VAR_PATTERN)) {
                return false;
            }
            // check if the env variables is $ENV
            if (!checkEnvVariableExistInText(text, systemVariablesMap, ENV_VAR_PATTERN_NO_BRACKETS)) {
                return false;
            }

            // check if the env variables is %ENV%
            if (!checkEnvVariableExistInText(text, systemVariablesMap, ENV_VAR_PATTERN_WINDOWS)) {
                return false;
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - substituteEnvVars - error while matching environment variable pattern"));
        }
        return true;
    }

    /**
     * This method encodes string on base 64
     *
     * @param str - the string to be encoded
     * @return encoded base 64 string
     */
    public static String base64Encoder(String str) {
        return Base64.getEncoder().encodeToString(str.getBytes());
    }

    /**
     * In this method we output the list of lines to the given file path.
     *
     * @param logContext - the log context
     * @param filePath   - the file path we want to write to
     * @param lines      - the list of lines we want to write to the file
     */
    public static void writeLinesToFile(LogContext logContext, Path filePath, List<String> lines) {
        try {
            Files.write(filePath, lines, StandardCharsets.UTF_8);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, PrivateRegistryUtils.class.getName() +
                    " - writeLinesToFile - error while writing lines to file "));
        }
    }

    /**
     * In this method we write the content of the Document object to the output file
     *
     * @param logContext - the log context
     * @param document   - the document holding the xml lines
     * @param filePath   - the file we want to write the lines to
     */
    public static void writeDocToXmlFile(LogContext logContext, Document document, String filePath) {
        try {
            String S_KEY_INDENT_AMOUNT = "{http://xml.apache.org/xalan}indent-amount";
            new XmlParser().removeEmptyText(document.getDocumentElement());
            FileWriter writer = new FileWriter(filePath);
            StreamResult result = new StreamResult(writer);
            TransformerFactory transformerFactory = TransformerFactory.newInstance();
            Transformer transformer = transformerFactory.newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty(OutputKeys.METHOD, "xml");
            transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
            transformer.setOutputProperty(S_KEY_INDENT_AMOUNT, "2");

            DOMSource source = new DOMSource(document);
            transformer.transform(source, result);
            writer.close();
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, PrivateRegistryUtils.class.getName() +
                    " - writeDocToXmlFile - error while writing to {} file "), Paths.get(filePath).getFileName());
        }
    }

    /**
     * This method writes the yaml dump to registry file
     *
     * @param registryFilePath - the registry file path we want to write to
     * @param yamlDump         - the yaml dump we want to write to the registry file
     * @throws FileNotFoundException
     */
    public static void writeYamlObjectToFile(String registryFilePath, Map<String, Object> yamlDump)
            throws FileNotFoundException {
        Yaml yaml = new Yaml();
        PrintWriter writer = new PrintWriter(registryFilePath);
        yaml.dump(yamlDump, writer);
        writer.close();
    }

    /**
     * In this method we retrieve a parent element (second level) which should be configured only once.
     * for example "servers" and "profiles" and "mirrors" and "active profiles"
     *
     * @param doc            - the document holding the xml lines
     * @param rootElement    - the root element of the document
     * @param elementTagName - the element we are looking for
     * @return the existing or created element
     */
    public static Element getParentElementFromDoc(Document doc, Element rootElement, String elementTagName) {
        Element element;
        if (doc.getElementsByTagName(elementTagName).item(0) == null) {
            element = doc.createElement(elementTagName);
            rootElement.appendChild(element);
        } else {
            element = (Element) doc.getElementsByTagName(elementTagName).item(0);
        }

        return element;
    }

    /**
     * remove the proceeding http/s from the match host and trailing "/"
     * @param matchHost - the full match host
     * @return the match host without http/s and trailing "/"
     */
    public static String getHostStrippedHttp(String matchHost) {
        String hostStrippedHttp = stripHttpHttps(matchHost);
        if (hostStrippedHttp.endsWith(Constants.FORWARD_SLASH)) {
            hostStrippedHttp = hostStrippedHttp.substring(0, hostStrippedHttp.length() - 1);
        }
        return hostStrippedHttp;
    }

    /**
     * In this method we extract the host name from the full match host
     *
     * @param matchHost - the full match host
     * @return the host name
     */
    public static String getHostName(String matchHost) {
        matchHost = stripHttpHttps(matchHost);
        int firstSlashInd = matchHost.contains(FORWARD_SLASH) ? matchHost.indexOf(FORWARD_SLASH) : matchHost.length();
        String host = matchHost.substring(0, firstSlashInd);
        return host;
    }


    /**
     * This method runs specific command and return the output as string
     * make sure the command doesn't include any ENV_VARS (e.g. '$HOME'),
     * and if required replace them with the var's value
     * (can be fetched from the map 'processBuilder.environment()'))
     *
     * @param logContext - the log context
     * @param command    - the command to run
     * @return the command output value
     */
    public static String runCommandAndReadOutput(LogContext logContext, List<String> command) {
        logger.info(LogUtils.formatLogMessage(logContext, "running the command: '{}'"),
                StringUtils.join(command, Constants.SPACE));
        ProcessBuilder processBuilder = new ProcessBuilder();
        processBuilder.command(command);
        String output = "";
        try {
            Process process = processBuilder.start();
            process.waitFor(30, TimeUnit.SECONDS);

            int exitVal = process.exitValue();
            logger.info(LogUtils.formatLogMessage(logContext, "Process exit value {}"), exitVal);
            if (exitVal == 0) {
                output = getOutput(process.getInputStream());
            } else {
                logger.error(LogUtils.formatLogMessage(logContext, getOutput(process.getErrorStream())));
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, PrivateRegistryUtils.class.getName() +
                            " - runCommandAndReadOutput - error while running the command {} "),
                    StringUtils.join(command, Constants.SPACE));
        }

        return output;
    }

    private static String getOutput(InputStream inputStream) throws IOException {
        StringBuilder buffer = new StringBuilder();
        BufferedReader in = new BufferedReader(
                new InputStreamReader(inputStream));
        String line;
        while ((line = in.readLine()) != null) {
            buffer.append(line);
        }
        return buffer.toString();
    }

    /* private methods */

    /**
     * In this method we run the matcher of the pattern on the text to find any env variables and checks if these
     * found env variables are located in the systemVariables map
     *
     * @param text               - the text we want to extract the env variables from
     * @param systemVariablesMap - the full system env variables map
     * @param pattern            - the specific pattern to look for the env variable
     * @return true if the env variables was located or no env variable was detected, or false if any env variable
     * was found, but it's not configured in the system variables map
     */
    private boolean checkEnvVariableExistInText(String text, Map<String, ?> systemVariablesMap, Pattern pattern) {
        Matcher matcher = pattern.matcher(text);
        while (matcher.find()) {
            String var = matcher.group(1);
            Object obj = systemVariablesMap.get(var);
            if (obj == null) {
                return false;
            }
        }
        return true;
    }

    /**
     * remove the proceeding http/s from the match host
     * @param matchHost - the full match host
     * @return the match host without http/s
     */
    private static String stripHttpHttps(String matchHost) {
        return matchHost.replace(HTTP_PATTERN, EMPTY_STRING).replace(HTTPS_PATTERN, EMPTY_STRING);
    }
}
