package com.wss.scanner.registry.scanners;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import org.apache.tools.ant.DirectoryScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

public class FilesScanner {
    /* --- Private Static Fields --- */
    private final static Logger logger = LoggerFactory.getLogger(FilesScanner.class);

    /* --- Private Fields --- */

    /* --- Constructors --- */
    public FilesScanner() {}

    /**
     * In this method we search for all the included manifest files under the project directory folder
     * @param logContext - the log context
     * @param projectDir - the project directory we are scanning
     * @param includes   - the files includes list to search for
     * @param excludes   - the files we want to exclude (pattern)
     * @return the found included manifest files under the project directory
     */
    public String[] searchForLocalManifestFiles(LogContext logContext, String projectDir, String[] includes, String[] excludes) {
        String[] files = new String[]{};

        try{
            files = searchDirForManifestFiles(logContext, projectDir, includes, excludes);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - searchForLocalManifestFiles - Failed walking the directory tree of {}"), projectDir);
        }

        return files;
    }

    /**
     * This method searches for each manifest file, for the related registry file (if found)
     * @param logContext             - the log context
     * @param manifestFile           - the found manifest file in the scanned directory
     * @param localRegistryFileNames - the local registry file names related to the manifest files list
     * @return the corresponding registry file of the specified manifest file
     */
    public String getExistingLocalRegistryFile(LogContext logContext, String manifestFile,
                                               List<String> localRegistryFileNames) {
        Path manifestFilePath = Paths.get(manifestFile);
        String parentPath = manifestFilePath.getParent().toString();
        for (String localRegistryFileName : localRegistryFileNames) {
            Path registryFilePath = Paths.get(parentPath, localRegistryFileName);
            if (registryFilePath.toFile().exists()) {
                logger.debug(LogUtils.formatLogMessage(logContext, "found registry file at {}"),
                        registryFilePath);
                return registryFilePath.toString();
            }
        }

        return null;
    }

    /**
     * This method searches for the manifest files specified in the includes pattern and excluding the ones under the
     * excludes pattern
     * @param logContext - the log context
     * @param scannedDir - the scanned directory
     * @param includes   - the includes files pattern list
     * @param excludes   - the excluded files pattern list
     * @return all the found manifest files following the includes pattern excluding the excludes pattern in the
     * scanned directory
     */
    private String[] searchDirForManifestFiles(LogContext logContext, String scannedDir, String[] includes,
                                               String[] excludes) {
        DirectoryScanner scanner = new DirectoryScanner();
        scanner.setBasedir(scannedDir);
        scanner.setIncludes(includes);
        scanner.setExcludes(excludes);
        scanner.setCaseSensitive(false);
        scanner.scan();
        return scanner.getIncludedFiles();
    }
}
