package com.wss.scanner.registry.utils.registryHandlers.python;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.scanner.registry.models.HostRule;
import com.wss.scanner.registry.utils.OsUtils;
import com.wss.scanner.registry.utils.PrivateRegistryUtils;
import com.wss.scanner.registry.utils.filesParsers.TextFileParser;
import com.wss.scanner.registry.utils.registryHandlers.PrivateRegistryFileHandler;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;

import static com.wss.scanner.registry.utils.Constants.*;

public class PythonPrivateRegistryHandler extends PrivateRegistryFileHandler {
    /* --- Private Static Fields --- */
    private final static Logger logger = LoggerFactory.getLogger(PythonPrivateRegistryHandler.class);

    static Boolean isPipEnvProject = false;

    /* --- Constructors --- */
    public PythonPrivateRegistryHandler() {
        // pipenv
    }

    /* --- Public Methods --- */
    @Override
    public void createRegistryObject(LogContext logContext, String registryEnvVariable, List<HostRule> hostRules) {
    }

    @Override
    public void editRegistryObject(LogContext logContext, String registryItem, List<HostRule> hostRules) {
        try {
            if (registryItem.endsWith(PIPFILE)) {
                isPipEnvProject = true;
                pipenvEditRegistryObject(logContext, hostRules, registryItem);
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - editRegistryObject - error while editing the env variable {} "), registryItem);
        }
    }

    @Override
    public boolean isProjectLevelRegistryFile() {
        return true;
    }

    @Override
    public boolean isSystemLevelRegistryFile() {
        return false;
    }

    @Override
    public boolean isEnvVariableRegistryCredentials() {
        return false;
    }

    @Override
    public void prepareIncludesAndExcludes(ArrayList<String> includes, ArrayList<String> excludes) {
        String[] pipenvtIncludedManifests = new String[]{
                PIPFILE
        };
        includes.addAll(Arrays.asList(pipenvtIncludedManifests));
    }

    @Override
    public List<String> getManifestTypes(String[] localManifestsFiles) {
        for (String localManifestFile : localManifestsFiles) {
            if (localManifestFile.equals(PIPFILE)) {
                return Collections.singletonList(localManifestFile);
            }
        }
        return new ArrayList();
    }

    @Override
    public List<String> getGlobalRegistryObject(boolean isFile) {
        return null;
    }

    @Override
    public List<String> getRegistryFileType(String manifestFile) {
        return Arrays.asList(PIPFILE);
    }



    /* --- Private Methods --- */

    private void pipenvEditRegistryObject(LogContext logContext, List<HostRule> hostRules, String registryItem) {
        logger.info(LogUtils.formatLogMessage(logContext, "Starting config the pipenv private repo"));
        List<String> lines = new LinkedList<>();
        List<String> custoumEnvVariables = new LinkedList<>();
        for (HostRule hostRule : hostRules) {
            logger.info(LogUtils.formatLogMessage(logContext, "Adding pipenv source configration section in Pipfile"));
            if (hostRule.getSourceName() != null) {
                String hostUrl = preparePipenvIndexUrl(logContext, hostRule);
                lines.add(NEW_LINE);
                lines.add(PIPENV_SOURCE);
                lines.add(String.format(URL + " = \"%s\" ", hostUrl));
                lines.add(PIPENV_ENABLE_SSL);
                lines.add(String.format(PIPENV_INDEX_NAME + "= \"%s\" ", hostRule.getSourceName()));
            }
            else if (hostRule.getEnvVariablesMapping() != null) {
                logger.info(LogUtils.formatLogMessage(logContext, "Adding pipenv custoum env variables"));
                if (hostRule.getEnvVariablesMapping().containsKey(PIPENV_ENV_PASSWARD) && hostRule.getPassword() != null) {
                    custoumEnvVariables.add(hostRule.getEnvVariablesMapping().get(PIPENV_ENV_PASSWARD) + EQUALS + hostRule.getPassword());
                    custoumEnvVariables.add(NEW_LINE);
                    custoumEnvVariables.add(hostRule.getEnvVariablesMapping().get(PIPENV_ENV_USER_NAME) + EQUALS + hostRule.getUserName());
                }
                else if  (hostRule.getEnvVariablesMapping().containsKey(PIPENV_ENV_TOKEN) && hostRule.getToken() != null) {
                    custoumEnvVariables.add(hostRule.getEnvVariablesMapping().get(PIPENV_ENV_TOKEN) + EQUALS + hostRule.getToken());
                    custoumEnvVariables.add(NEW_LINE);
                    if (hostRule.getEnvVariablesMapping().containsKey(PIPENV_ENV_USER_NAME) && hostRule.getUserName() != null) {
                        custoumEnvVariables.add(NEW_LINE);
                        custoumEnvVariables.add(hostRule.getEnvVariablesMapping().get(PIPENV_ENV_USER_NAME) + EQUALS + hostRule.getUserName());
                    }
                }
                else{
                    logger.warn(LogUtils.formatLogMessage(logContext, "No password or token were found to create pipenv env variable"));
                }
            }
            else {
                logger.warn(LogUtils.formatLogMessage(logContext, "No sourceName or envVariablesMapping were found"));
            }

        }
        if (!lines.isEmpty()) {
            createPipEnvSourceBlock(logContext, registryItem, lines);
        }
        if (!custoumEnvVariables.isEmpty()) {
            try {
                createCustoumEnvVariablesFile(logContext, registryItem, custoumEnvVariables);
            } catch (IOException ex) {
                logger.error(LogUtils.getExceptionErrorMessage(logContext, ex, this.getClass().getName() +
                        " - Error while Create Pipev env file "));
            }
        }
    }

    private String preparePipenvIndexUrl(LogContext logContext, HostRule hostRule) {
        logger.info(LogUtils.formatLogMessage(logContext, "Building Pipenv Index Url"));
        String userName = (StringUtils.isNotBlank(hostRule.getUserName()) && StringUtils.isNotBlank(hostRule.getPassword())) ? hostRule.getUserName() : "";
        String tokenOrPassword;
        if (StringUtils.isNotBlank(hostRule.getToken())) {
            tokenOrPassword = hostRule.getToken();
        } else if (StringUtils.isNotBlank(hostRule.getPassword())) {
            tokenOrPassword = hostRule.getPassword();
        } else {
            tokenOrPassword = "";
            logger.warn(LogUtils.formatLogMessage(logContext, "No password or token were found"));
            return null;
        }

        try {
            tokenOrPassword = URLEncoder.encode(tokenOrPassword, StandardCharsets.UTF_8.toString());
        } catch (UnsupportedEncodingException e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " error while encoding the password "));
        }
        tokenOrPassword = StringUtils.isNotBlank(userName) ?  userName + COLON + tokenOrPassword + AT :tokenOrPassword + AT;
        String matchHost = hostRule.getMatchHost();
        String urlWithHttpPattern = tokenOrPassword + matchHost.replace(HTTP_PATTERN, EMPTY_STRING).replace(HTTPS_PATTERN, EMPTY_STRING);
        String hostWithCredentials = matchHost.startsWith(HTTP_PATTERN) ? HTTP_PATTERN + urlWithHttpPattern : HTTPS_PATTERN + urlWithHttpPattern;

        if (!hostWithCredentials.endsWith(FORWARD_SLASH + SIMPLE) &&
                !hostWithCredentials.endsWith(FORWARD_SLASH + SIMPLE + FORWARD_SLASH)) {
            hostWithCredentials = hostWithCredentials.endsWith(FORWARD_SLASH) ? hostWithCredentials.concat(SIMPLE) :
                    hostWithCredentials.concat(FORWARD_SLASH + SIMPLE);
        }
        return hostWithCredentials;

    }

    private void createPipEnvSourceBlock(LogContext logContext, String registryItem, List<String> lines) {
        TextFileParser textFileParser = new TextFileParser();
        List<String> existingLines = textFileParser.parseTxtFileLineByLine(logContext, registryItem);
        Path path = Paths.get(registryItem);
        existingLines.addAll(lines);
        PrivateRegistryUtils.writeLinesToFile(logContext, path, existingLines);
    }

    private void createCustoumEnvVariablesFile(LogContext logContext, String registryItem, List<String> custoumEnvVariables) throws IOException {
        Path envFilePath = Paths.get(registryItem.split(PIPFILE)[0] + PIPENV_ENV_FILE_NAME);
        envFilePath.toFile().createNewFile();
        PrivateRegistryUtils.writeLinesToFile(logContext, envFilePath, custoumEnvVariables);
    }
}
