package com.wss.scanner.registry;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.common.results.scaResultsDTO.ScaResObjDTO;
import com.wss.common.results.scaResultsDTO.ScaResultsDTO;
import com.wss.common.results.enums.*;
import com.wss.scanner.registry.models.HostRule;
import com.wss.scanner.registry.models.PSBResult;
import com.wss.scanner.registry.scanners.FilesScanner;
import com.wss.scanner.registry.utils.registryHandlers.PrivateRegistryFileHandler;
import com.wss.scanner.registry.utils.registryHandlers.python.PythonPrivateRegistryHandler;
import com.wss.scanner.registry.utils.registryHandlers.sbt.SbtPrivateRegistryHandler;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.*;

import static com.wss.scanner.registry.utils.Constants.*;

public class PrivateRegistriesHandler {

    /* --- Private Static Fields --- */
    private final static Logger logger = LoggerFactory.getLogger(PrivateRegistriesHandler.class);

    /* --- Private Fields --- */
    @Getter @Setter
    private PSBResult psbResult;
    private final Map<String, String> registryFileToOriginPathHistory;


    /* --- Constructors --- */
    public PrivateRegistriesHandler() {
        this.registryFileToOriginPathHistory = new HashMap<>();
    }

    /**
     * This method is handling all the private registry files search/read/create/edit
     *
     * @param logContext                      - the log context
     * @param projectFolder                   - the project folder
     * @param hostRules                       - the provided host rules map
     * @param requestOrigin                   - the request origin object
     * @param registryFileToOriginPathHistory - the map holding the created/edited registry files
     */
    public void handlePrivateRegistries(LogContext logContext, String projectFolder, List<HostRule> hostRules,
                                        String requestOrigin, Map<String, String> registryFileToOriginPathHistory) throws Exception {
        this.registryFileToOriginPathHistory.putAll(registryFileToOriginPathHistory);
        // extract the used resolvers, and create an object of each resolver type to the relevant hostRuleDTO
        Map<String, Map<String, List<HostRule>>> resolverTypeToHostRules = mapHostRulesToResolversTypes(hostRules);

        handleResolversWithLocalRegistryFiles(logContext, projectFolder, resolverTypeToHostRules.get(PROJECT_LEVEL),
                requestOrigin);
        handleResolversWithGlobalRegistryFiles(logContext, projectFolder, resolverTypeToHostRules.get(SYSTEM_LEVEL),
                requestOrigin);
        handleEnvVariableRegistryCredentials(logContext, projectFolder, resolverTypeToHostRules.get(ENV_VARIABLE_LEVEL));
    }

    /**
     * This method is for restoring the registry files that were edited or removing the ones that was created
     *
     * @param logContext - the log context
     *                   registryFileToOriginPathHistory - the map holding the created/edited registry files
     */
    public void restorePrivateRegistriesFiles(LogContext logContext) throws Exception {
        try {
            for (Map.Entry<String, String> registryFilePath : registryFileToOriginPathHistory.entrySet()) {
                String originFile = registryFilePath.getValue(); // the registry file origin
                String newFile = registryFilePath.getKey(); // the edited/created registry file
                if (originFile == null) {
                    // this mean that it's a newly created file which was not exists
                    File registryFile = new File(newFile);
                    if (registryFile.exists()) {
                        FileUtils.forceDelete(registryFile);
                    }
                } else {
                    File registryFile = new File(originFile);
                    if (registryFile.exists()) {
                        // first copy the content of the origin file to the edited file, and then remove the copied
                        // origin file
                        Files.copy(Paths.get(originFile), Paths.get(newFile), StandardCopyOption.REPLACE_EXISTING);
                        FileUtils.forceDelete(registryFile);
                    }
                }
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - restorePrivateRegistriesFiles - error while handling private registries "));
        }
    }

    /**
     * In this method we map each resolver type to the corresponding HostRule objects
     *
     * @param hostRules - the original full host rules list
     * @return a map with resolver type as key and all the related HostRule objects as a value (list)
     */
    private Map<String, Map<String, List<HostRule>>> mapHostRulesToResolversTypes(List<HostRule> hostRules) {
        Map<String, Map<String, List<HostRule>>> resolutionTypeToResolversHostRules = new HashMap<>();
        resolutionTypeToResolversHostRules.put(SYSTEM_LEVEL, new HashMap<>());
        resolutionTypeToResolversHostRules.put(PROJECT_LEVEL, new HashMap<>());
        resolutionTypeToResolversHostRules.put(ENV_VARIABLE_LEVEL, new HashMap<>());

        Map<String, List<HostRule>> hostRulesTypes = new HashMap<>();
        for (HostRule hostRule : hostRules) {
            String hostType = hostRule.getHostType();
            List<HostRule> hostRulesList;
            if (hostRulesTypes.containsKey(hostType)) {
                hostRulesList = hostRulesTypes.get(hostType);
                hostRulesList.add(hostRule);
            } else {
                hostRulesList = new LinkedList<>();
                hostRulesList.add(hostRule);
                hostRulesTypes.put(hostType, hostRulesList);
            }
        }

        for (Map.Entry<String, List<HostRule>> hostRuleTypeEntity : hostRulesTypes.entrySet()) {
            String resolverType = hostRuleTypeEntity.getKey();
            PrivateRegistryFileHandler resolverHandler = getResolverPrivateRegistryHandler(resolverType);
            if (resolverHandler != null) {
                if (resolverHandler.isSystemLevelRegistryFile()) {
                    Map<String, List<HostRule>> resolverToHostRulesMap = resolutionTypeToResolversHostRules.get(SYSTEM_LEVEL);
                    resolverToHostRulesMap.put(resolverType, hostRuleTypeEntity.getValue());
                }
                if (resolverHandler.isProjectLevelRegistryFile()) {
                    Map<String, List<HostRule>> resolverToHostRulesMap = resolutionTypeToResolversHostRules.get(PROJECT_LEVEL);
                    resolverToHostRulesMap.put(resolverType, hostRuleTypeEntity.getValue());
                }
                if (resolverHandler.isEnvVariableRegistryCredentials()) {
                    Map<String, List<HostRule>> resolverToHostRulesMap = resolutionTypeToResolversHostRules.get(ENV_VARIABLE_LEVEL);
                    resolverToHostRulesMap.put(resolverType, hostRuleTypeEntity.getValue());
                }

            }
        }

        return resolutionTypeToResolversHostRules;
    }

    /**
     * This method is to handle all the resolvers that uses local registry files (registry files that are in the project
     * level, for example .npmrc will be put besides the package.json file)
     *
     * @param logContext              - the log context
     * @param projectFolder           - the scanned project folder
     * @param resolverTypeToHostRules - the map of resolvers types and corresponding host rules
     * @param requestOrigin           - the request origin object
     *                                registryFileToOriginPathHistory - a map holding the registry files history
     */
    private void handleResolversWithLocalRegistryFiles(LogContext logContext, String projectFolder,
                                                       Map<String, List<HostRule>> resolverTypeToHostRules,
                                                       String requestOrigin) {
        if (resolverTypeToHostRules.isEmpty()) {
            logger.info(LogUtils.formatLogMessage(logContext, "No resolver with local registry " +
                    "handling was found"));
            return;
        }

        FilesScanner filesScanner = new FilesScanner();
        ArrayList<String> includes = new ArrayList<>();
        ArrayList<String> excludes = new ArrayList<>();
        prepareFullIncludesExcludes(resolverTypeToHostRules, includes, excludes);

        /*
            the local manifests files will hold a map having the resolver type as key, and a map of all the related
            discovered local manifests files with the related registry files, for example:
            manifestFilesTypes = {
                npm: {
                    dir1/package.json: dir1/.npmrc,
                    dir2/package.json: null
                }
            }
         */
        Map<String, Map<String, String>> manifestFilesTypes = mapLocalManifestsFilesToRegistryFiles(logContext,
                projectFolder, includes, excludes, resolverTypeToHostRules.keySet(), filesScanner);

        editCreateLocalRegistryFiles(logContext, resolverTypeToHostRules, manifestFilesTypes, requestOrigin);
    }

    /**
     * This method is to handle all the resolvers that uses global registry files (registry files that are in the user
     * level, for example settings.xml)
     *
     * @param logContext                    - the log context
     * @param projectFolder                 - the scanned project folder
     * @param resolverTypeToGlobalHostRules - the map of resolvers types and corresponding host rules
     * @param requestOrigin                 - the request origin object
     *                                      registryFileToOriginPathHistory - a map holding the registry files history
     */
    private void handleResolversWithGlobalRegistryFiles(LogContext logContext, String projectFolder,
                                                        Map<String, List<HostRule>> resolverTypeToGlobalHostRules,
                                                        String requestOrigin) {
        if (resolverTypeToGlobalHostRules.isEmpty()) {
            logger.info(LogUtils.formatLogMessage(logContext, "No resolver with global registry " +
                    "handling was found"));
            return;
        }

        editCreateGlobalRegistryFiles(logContext, projectFolder, resolverTypeToGlobalHostRules, requestOrigin);
    }

    /**
     * This method is to handle all the resolvers that uses registry env variables configurations
     *
     * @param logContext                         - the log context
     * @param envVariableResolverTypeToHostRules - the map of env variables resolvers and the corresponding host rules
     */
    private void handleEnvVariableRegistryCredentials(LogContext logContext, String projectFolder,
                                                      Map<String, List<HostRule>> envVariableResolverTypeToHostRules) {
        if (envVariableResolverTypeToHostRules.isEmpty()) {
            logger.info(LogUtils.formatLogMessage(logContext, "No resolver with environment variables credentials " +
                    "handling was found"));
            return;
        }

        editCreateEnvVariables(logContext, projectFolder, envVariableResolverTypeToHostRules);
    }

    /**
     * This method create or edit the needed env variables values based on the host rules map in order to connect
     * to the private registry of the specific resolver
     *
     * @param logContext                         - the log context
     * @param envVariableResolverTypeToHostRules - the map of env variables resolvers and the corresponding host rules
     */
    private void editCreateEnvVariables(LogContext logContext, String projectFolder,
                                        Map<String, List<HostRule>> envVariableResolverTypeToHostRules) {
        for (String resolverType : envVariableResolverTypeToHostRules.keySet()) {
            // get the resolver handler
            PrivateRegistryFileHandler privateRegistryHandler = getResolverPrivateRegistryHandler(resolverType);

            if (privateRegistryHandler != null) {
                List<HostRule> hostRules = envVariableResolverTypeToHostRules.get(resolverType);
                List<String> envVariables = privateRegistryHandler.getGlobalRegistryObject(false);
                Map<String, String> systemEnvVariables = System.getenv();

                for (String envVariable : envVariables) {
                    if (systemEnvVariables.containsKey(envVariable)) {
                        logger.debug(LogUtils.formatLogMessage(logContext, "Environment variable '{}' is already " +
                                "configured, editing it."), envVariable);
                        privateRegistryHandler.editRegistryObject(logContext, envVariable, hostRules);
                    } else {
                        logger.debug(LogUtils.formatLogMessage(logContext, "Environment variable '{}' is not " +
                                "configured, creating it."), envVariable);
                        privateRegistryHandler.createRegistryObject(logContext, envVariable, hostRules);
                    }
                }

            }
        }
    }

    /**
     * In this method we check if we already have global registry files, then we edit them, or create new global
     * registry files if needed
     *
     * @param logContext                    - the log context
     * @param projectFolder                 - the folder of the scanned project
     * @param resolverTypeToGlobalHostRules - the map of holding the global resolvers and its corresponding host rules
     * @param requestOrigin                 - the request origin object
     */
    private void editCreateGlobalRegistryFiles(LogContext logContext, String projectFolder,
                                               Map<String, List<HostRule>> resolverTypeToGlobalHostRules,
                                               String requestOrigin) {
        for (String resolverType : resolverTypeToGlobalHostRules.keySet()) {
            // get the resolver handler
            PrivateRegistryFileHandler privateRegistryHandler = getResolverPrivateRegistryHandler(resolverType);
            if (privateRegistryHandler != null) {
                List<String> globalRegistryObjects = privateRegistryHandler.getGlobalRegistryObject(true);
                if (!globalRegistryObjects.isEmpty()) {
                    for (String globalRegistryFile : globalRegistryObjects) {
                        if (new File(globalRegistryFile).exists()) {
                            // registry file exists (for example settings.xml), we need to edit it
                            logger.debug(LogUtils.formatLogMessage(logContext, "Found global registry file at {} for " +
                                    "{} resolver. editing it."), globalRegistryFile, resolverType);
                            copyOriginRegistryFile(logContext, requestOrigin, globalRegistryFile, true);
                            privateRegistryHandler.editRegistryObject(logContext, globalRegistryFile,
                                    resolverTypeToGlobalHostRules.get(resolverType));
                        } else {
                            // registry file is missing, create a new one
                            List<String> registryFileSuffixes = privateRegistryHandler.getRegistryFileType(null);
                            for (String registryFileSuffix : registryFileSuffixes) {
                                logger.debug(LogUtils.formatLogMessage(logContext, "No global registry file was found at {} " +
                                                "for {} resolver. creating a new {} registry file."),
                                        globalRegistryFile, resolverType, registryFileSuffix);
                                privateRegistryHandler.createRegistryObject(logContext, globalRegistryFile,
                                        resolverTypeToGlobalHostRules.get(resolverType));
                                registryFileToOriginPathHistory.put(globalRegistryFile, null);
                            }
                        }
                    }
                }
            }
        }
    }


    /**
     * In this method we check if we already have the registry file, then we edit it, or create a new registry file
     * if it doesn't exist
     *
     * @param logContext         - the log context
     * @param hostRulesTypes     - the host rules types map
     * @param manifestFilesTypes - the object mapping resolvers to manifest/registry files
     * @param requestOrigin      - the request origin object
     */
    private void editCreateLocalRegistryFiles(LogContext logContext, Map<String, List<HostRule>> hostRulesTypes,
                                              Map<String, Map<String, String>> manifestFilesTypes, String requestOrigin) {
        for (String resolverType : manifestFilesTypes.keySet()) {
            // get the resolver handler
            PrivateRegistryFileHandler privateRegistryHandler = getResolverPrivateRegistryHandler(resolverType);

            if (privateRegistryHandler != null) {
                Map<String, String> manifestsFilesMap = manifestFilesTypes.get(resolverType);
                for (String manifestFile : manifestsFilesMap.keySet()) {
                    String registryFile = manifestsFilesMap.get(manifestFile);
                    String manifestFileFolder = Paths.get(manifestFile).getParent().toString();
                    if (registryFile != null) {
                        // registry file exist - edit it
                        logger.debug(LogUtils.formatLogMessage(logContext, "Found local registry file at {} for " +
                                "{} resolver. editing it."), registryFile, resolverType);
                        copyOriginRegistryFile(logContext, requestOrigin, registryFile, false);
                        privateRegistryHandler.editRegistryObject(logContext, registryFile, hostRulesTypes.get(resolverType));
                    } else {
                        // no registry file was found near the manifest file - create a new registry file
                        List<String> registryFileNames = privateRegistryHandler.getRegistryFileType(manifestFile);
                        for (String registryFileName : registryFileNames) {
                            logger.debug(LogUtils.formatLogMessage(logContext, "No registry file was found near the " +
                                            "manifest file {} for {} resolver. creating a new {} registry file."),
                                    manifestFile, resolverType, registryFileName);
                            String registryFilePath = Paths.get(manifestFileFolder, registryFileName).toString();
                            privateRegistryHandler.createRegistryObject(logContext, registryFilePath, hostRulesTypes.get(resolverType));
                            registryFileToOriginPathHistory.put(registryFilePath, null);
                        }
                    }
                }
            }
        }
    }

    /**
     * In this method we copy the registry file to a temp file ending with "_origin" so we can restore its content
     * from this copy
     *
     * @param logContext    - the log context
     * @param requestOrigin - the request origin
     * @param registryFile  - the registry file we want to copy
     */
    private void copyOriginRegistryFile(LogContext logContext, String requestOrigin, String registryFile, boolean isGlobal) {
        try {
            if (requestOrigin.contains(OP) || isGlobal) {
                Path registryFilePath = Paths.get(registryFile);
                Path originRegistryFile = Paths.get(registryFilePath.toString() + ORIGIN);
                Files.copy(registryFilePath, originRegistryFile, StandardCopyOption.REPLACE_EXISTING);
                registryFileToOriginPathHistory.put(registryFilePath.toString(), originRegistryFile.toString());
            }
        } catch (IOException e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - copyOriginRegistryFile - error while coping file {}"), registryFile);
        }
    }

    /**
     * This method find all the manifests files (following the includes/excludes) and map each one of them to the
     * relevant resolver type
     *
     * @param logContext           - the log context
     * @param projectFolder        - the project folder to scan
     * @param includes             - the files patterns we wish to include in the scan
     * @param excludes             - the files patterns we wish to exclude from the scan
     * @param inUseResolversToScan - the found in use resolver to scan locally
     * @param filesScanner         - the filesScanner class object
     * @return a map with each resolver and It's corresponding to found manifest files.
     */
    private Map<String, Map<String, String>> mapLocalManifestsFilesToRegistryFiles(LogContext logContext, String projectFolder,
                                                                                   ArrayList<String> includes,
                                                                                   ArrayList<String> excludes,
                                                                                   Set<String> inUseResolversToScan,
                                                                                   FilesScanner filesScanner) {
        Map<String, Map<String, String>> manifestFilesTypes = new HashMap<>();
        String[] localManifestsFiles = filesScanner.searchForLocalManifestFiles(logContext, projectFolder,
                includes.toArray(new String[0]), excludes.toArray(new String[0]));

        // loop over all the found manifests files
        for (String manifestFile : localManifestsFiles) {
            String manifestFileSuffix = Paths.get(manifestFile).getFileName().toString(); // for example "package.json"
            // loop over the used resolvers only
            for (String resolverType : inUseResolversToScan) {
                //get the handler of the resolver
                PrivateRegistryFileHandler resolverHandler = getResolverPrivateRegistryHandler(resolverType);

                if (resolverHandler != null && resolverHandler.getManifestTypes(localManifestsFiles)
                        .stream().anyMatch(s -> s.endsWith(manifestFileSuffix))) {
                    Map<String, String> manifestFileToRegistryFile = manifestFilesTypes.get(resolverType);
                    String manifestFilePath = Paths.get(projectFolder, manifestFile).toString();
                    String registryFilePath = filesScanner.getExistingLocalRegistryFile(logContext, manifestFilePath,
                            resolverHandler.getRegistryFileType(manifestFile));

                    if (manifestFileToRegistryFile == null) {
                        manifestFileToRegistryFile = new HashMap<>();
                        manifestFilesTypes.put(resolverType, manifestFileToRegistryFile);
                    }
                    manifestFileToRegistryFile.put(manifestFilePath, registryFilePath);
                }
            }
        }
        return manifestFilesTypes;
    }

    /**
     * This method gets the provided host rules and build the full includes/excludes patterns
     *
     * @param hostRulesTypes - the map of resolver type and the relevant provided host rule
     * @param includes       - the includes patterns full list
     * @param excludes       - the excludes patterns full list
     */
    private void prepareFullIncludesExcludes(Map<String, List<HostRule>> hostRulesTypes, ArrayList<String> includes,
                                             ArrayList<String> excludes) {
        for (String resolverType : hostRulesTypes.keySet()) {
            PrivateRegistryFileHandler resolverHandler = getResolverPrivateRegistryHandler(resolverType);
            if (resolverHandler != null) {
                resolverHandler.prepareIncludesAndExcludes(includes, excludes);
            }
        }
    }

    /**
     * In this method will we get the suitable private registry handler class
     *
     * @param resolverType - the resolver type we want to get the suitable handler class for
     * @return the private registry handler class
     */
    private PrivateRegistryFileHandler getResolverPrivateRegistryHandler(String resolverType) {
        if (resolverType.equals(SBT)) {
            return new SbtPrivateRegistryHandler();
        }
        return null;
    }

    public void setRegistryFilesToRestore(ScaResultsDTO scaResults) {
        scaResults.getResults().forEach((path, pathResults) -> {
            for (ScaResObjDTO resObj : pathResults) {
                if (resObj.getTool().equals(ToolEnum.PSB.toString()) && resObj.getStage().equals(StageEnum.CONFIG.toString())) {
                    try {
                        Path originRegistryFile = Paths.get(path + ORIGIN);
                        if (Files.exists(originRegistryFile))
                            registryFileToOriginPathHistory.put(path, originRegistryFile.toString());
                    } catch (Exception ignored) {
                    }
                    break;
                }
            }
        });
    }
}
