TYPE_IMPORT_ALL = "IMPORT_ALL"
TYPE_IMPORT = "IMPORT"
TYPE_FILE = "FILE"
TYPE_METHOD = "METHOD"
TYPE_CLASS = "CLASS"
TYPE_IDENTIFIER = "IDENTIFIER"
TYPE_COMPOSITE = "COMPOSITE"
TYPE_ASSIGNMENT = "ASSIGNMENT"
TYPE_DEFAULT_ASSIGNMENT = "DEFAULT_ASSIGNMENT"
TYPE_LITERAL = "LITERAL"
TYPE_RETURN = "RETURN"
TYPE_METHOD_INVOCATION = "METHOD_INVOCATION"
TYPE_MEMBER_EXPRESSION = "MEMBER_EXPRESSION"
TYPE_UNPACK_ARG = "UNPACK_ARG"


class ARNode(object):

    def __init__(self, node_type):
        super(ARNode,self).__init__()
        self.node_type = node_type

    def add_position(self, row, col):
        self.position = Position(row, col)
        for child in self.get_children():
            if not hasattr(child, "position") or child.position is None:
                child.add_position(row, col)

    def get_children(self):
        res = []
        fields = [getattr(self, field_name) for field_name in dir(self) if not field_name.startswith("__")]
        for field in fields:
            if isinstance(field, list):
                for item in field:
                    if isinstance(item, ARNode):
                        res.append(item)
            elif isinstance(field, ARNode):
                res.append(field)
        return res


class Type(ARNode):

    def __init__(self, name, node_type):
        super(Type,self).__init__(node_type)
        self.cfg = []
        self.declarations = []
        self.name = name

    def extend_with_context(self, context):
        self.cfg.extend(context.cfg)
        self.cfg.extend(context.results)
        self.declarations.extend(context.declarations)


class File(Type):

    def __init__(self, filepath, name):
        super(File, self).__init__(name, TYPE_FILE)
        self.filepath = filepath


class Method(Type):

    def __init__(self, name, parameters, is_static=False, vararg=None, kwarg=None):
        super(Method,self).__init__(name, TYPE_METHOD)
        self.parameters = parameters
        self.varargs = vararg
        self.kwargs = kwarg
        self.isStatic = is_static


class Class(Type):

    def __init__(self, name, superclass_expressions):
        super(Class, self).__init__(name, TYPE_CLASS)
        self.superclass_expressions = superclass_expressions


class Expression(ARNode):

    def __init__(self, node_type):
        super(Expression,self).__init__(node_type)


class Identifier(Expression):

    def __init__(self, name):
        super(Identifier,self).__init__(TYPE_IDENTIFIER)
        self.name = name


class Literal(Expression):
    LITERAL_FLOAT = "FLOAT"
    LITERAL_INT = "INT"
    LITERAL_BOOL = "BOOL"
    LITERAL_STRING = "STR"
    LITERAL_NONE = "NONE"
    LITERAL_BYTES = "BYTES"

    def __init__(self, literal_type, value):
        super(Literal,self).__init__(TYPE_LITERAL)
        self.literalType = literal_type
        self.value = value


class Return(Expression):

    def __init__(self, expression):
        super(Return,self).__init__(TYPE_RETURN)
        self.expression = expression


class MethodInvocation(Expression):

    def __init__(self, target, arguments, named_parameters=None):
        super(MethodInvocation,self).__init__(TYPE_METHOD_INVOCATION)
        if named_parameters is None:
            named_parameters = []
        self.named_parameters = named_parameters
        self.target = target
        self.arguments = arguments


class Assignment(ARNode):

    def __init__(self, lhs, rhs):
        super(Assignment,self).__init__(TYPE_ASSIGNMENT)
        self.lhs = lhs
        self.rhs = rhs


class DefaultAssignment(Assignment):

    def __init__(self, lhs, rhs):
        super(DefaultAssignment,self).__init__(lhs, rhs)
        self.node_type = TYPE_DEFAULT_ASSIGNMENT


class MemberExpression(Expression):

    def __init__(self, object, member):
        super(MemberExpression,self).__init__(TYPE_MEMBER_EXPRESSION)
        self.object = object
        self.member = member


class Import(ARNode):

    def __init__(self, name, filepath):
        super(Import,self).__init__(TYPE_IMPORT)
        self.name = name
        self.filepath = filepath


class ImportAll(ARNode):

    def __init__(self, name, filepath):
        super(ImportAll,self).__init__(TYPE_IMPORT_ALL)
        self.name = name
        self.filepath = filepath


class UnpackArgument(ARNode):
    def __init__(self, exp):
        super(UnpackArgument,self).__init__(TYPE_UNPACK_ARG)
        self.exp = exp


class Position:
    def __init__(self, line, column):
        self.line = line
        self.column = column
