from __future__ import print_function

import ntpath
import os
import sys

from sources.imprecise_analysis.FileParser import get_imports
from sources.imprecise_analysis.FileParser import get_objects

PY_EXTENSION = ".py"
INIT_FILE = "__init__.py"


def is_single_file_module(dir_module):
    return not os.path.isdir(dir_module) and os.path.isfile(
        dir_module + PY_EXTENSION)


def get_relative_filename(dir_module, filename, module_name):
    if is_single_file_module(dir_module):
        return module_name + PY_EXTENSION
    result = filename.replace(dir_module, "")
    if result.startswith(os.sep):
        result=result[1:]
    return result


def get_relative_dots(import_level):
    dots = ''
    for i in range(import_level):
        dots += '.'
    return dots


def remove_leading_dots(module):
    count = 0
    while module.startswith("."):
        module = module[1:]
        count = count + 1
    return module, count


def to_init_file_if_folder(external_folder):
    if os.path.isdir(external_folder):
        init_file_path = os.path.join(external_folder, INIT_FILE)
        return init_file_path
    return external_folder


def get_relative_path(import_level):
    dots = ''
    if import_level == 0:
        level = 0
    else:
        level = import_level - 1
    for i in range(level):
        dots += '../'
    return dots


def remove_from_end(string, ending):
    if string.endswith(ending):
        idx = string.rfind(ending)
        return string[:idx]
    return string


class ModuleParser:
    def __init__(self, site_packages):
        self.site_packages = site_packages
        self.visited = []
        self.visited_files = []
        self.dots = {}
        self.imports = {}
        self.working_list = []
        self.init_file_to_objects = {}
        self.app_parse_errors = False
        self.ext_parse_errors = False
        self.errors = False

    def analyze_transitive_dependencies(self, dir_module, log_dir):
        if not os.path.isdir(dir_module):
            pass
        self.visited.append(dir_module)
        module_files = self.get_files_from_module(dir_module)
        self.working_list = module_files
        while len(self.working_list) > 0:
            file_source, mdl_path = self.working_list.pop()
            if file_source in self.visited_files:
                continue
            self.visited_files.append(file_source)
            try:
                module_to_level_and_names = get_imports(file_source)
            except (SyntaxError, IOError) as e:
                print("Failed parsing file " + file_source + ". Err=" + str(e), file=sys.stderr)
                self.log_parse_fail(file_source, log_dir)
                if dir_module == mdl_path:
                    self.app_parse_errors = True
                else:
                    self.ext_parse_errors = True
                continue
            except Exception as e:
                print("Failed parsing file " + file_source + ". Err=" + str(e), file=sys.stderr)
                self.errors = True
                continue
            modules = module_to_level_and_names.keys()
            data_to_insert = set()
            for module in modules:
                level_and_names = module_to_level_and_names[module]
                level = level_and_names[0][0]
                imported_full_paths_with_names, external_mod = self.get_full_path(module, mdl_path, file_source, level,
                                                                                  level_and_names)
                if external_mod == '':
                    external_mod = mdl_path  # internal

                for ext_file in imported_full_paths_with_names:
                    if ext_file is not None:
                        self.working_list.append((ext_file, external_mod))

                for imported_full_path in imported_full_paths_with_names:
                    names = imported_full_paths_with_names[imported_full_path]
                    for name in names:
                        self.work_on_file(data_to_insert, mdl_path, file_source,
                                          level, imported_full_path,
                                          module, name, external_mod)
            if mdl_path.endswith(os.sep):
                basename = os.path.basename(mdl_path[:-1])
            else:
                basename = os.path.basename(mdl_path)
            if basename not in self.dots:
                self.dots[basename] = {}
            module_to_imported = self.dots[basename]
            if len(data_to_insert) > 0:
                key_file_name = get_relative_filename(mdl_path, file_source,
                                                      basename)
                module_to_imported[key_file_name] = data_to_insert
        return self.dots

    def log_parse_fail(self, file_source, log_dir):
        if os.path.exists(log_dir):
            with open(os.path.join(log_dir, "parse_error.log"), "a") as log_file:
                log_file.write(file_source + os.linesep)

    def get_files_from_module(self, dir_module):
        files = []
        for root, directories, filenames in os.walk(dir_module):
            for filename in filenames:
                file_name = os.path.join(root, filename)
                if file_name.endswith(PY_EXTENSION):
                    files.append((file_name, dir_module))
        return files

    def work_on_file(self, data_to_insert, dir_module, filename, import_level, imported_full_path, module,
                     import_name, external_module):
        if self.is_package_from_import(import_level, import_name, imported_full_path):
            package_init_file = os.path.join(os.path.dirname(filename), INIT_FILE)
            if os.path.exists(package_init_file):
                imported_full_path = package_init_file
        if imported_full_path and imported_full_path != filename:
            relative_mdl = get_relative_dots(import_level) + module
            if import_name:
                relative_mdl = relative_mdl + import_name
            self.add_graph_arc(data_to_insert, dir_module, filename, imported_full_path, relative_mdl)
            if not filename.startswith(os.path.dirname(imported_full_path)):
                self.add_partial_imports(data_to_insert, dir_module, filename, imported_full_path, relative_mdl,
                                         import_name, external_module)

    def is_package_from_import(self, import_level, import_name, imported_full_path):
        return imported_full_path is None and import_name is None and import_level > 0

    def add_graph_arc(self, data_to_insert, dir_module, filename, imported_full_path, relative_mdl):
        self.imports[filename + ":" + relative_mdl] = imported_full_path
        graph_val = self.get_destination(imported_full_path, dir_module, relative_mdl)
        data_to_insert.add(graph_val)

    def add_partial_imports(self, data_to_insert, dir_module, filename, imported_full_path, module, import_name,
                            external_module):
        if import_name:
            module = module[:-len(import_name)]
            module = module + "." + import_name
        module, num_leading_dots = remove_leading_dots(module)
        module_parts = module.split(".")
        if self.is_submodule_import(imported_full_path, module_parts):
            module = self.remove_last_module(module)
            module_parts = module_parts[:-1]
        if not imported_full_path.endswith(INIT_FILE):
            module = self.remove_last_module(module)
            module_parts = module_parts[:-1]

        if module_parts:
            for _ in reversed(module_parts):
                imported_full_path = os.path.dirname(imported_full_path)
                if filename.startswith(imported_full_path):
                    break
                init_file = os.path.join(imported_full_path, INIT_FILE)
                if os.path.isfile(init_file):
                    module_with_relative = "." * num_leading_dots + module
                    self.add_graph_arc(data_to_insert, dir_module, filename, init_file, module_with_relative)
                    if init_file not in self.working_list:
                        self.working_list.append((init_file, external_module))
                module = self.remove_last_module(module)

    def remove_last_module(self, module):
        last_dot_idx = module.rfind(".")
        if last_dot_idx == -1:
            return ""
        return module[:last_dot_idx]

    def is_submodule_import(self, imported_full_path, module_parts):
        """
        :param imported_full_path: the imported target filepath
        :param module_parts: a list of the imported parts that originated this import, i.e.,
        the imported name parts from an Import statement (import a.b.c ~> [a,b,c]) or the from name parts and the
        import name from a FromImport statement (from a.b import c ~> [a,b,c])
        :return: True if the imported object was inside a module (and False if it was a module)
        """
        imported_full_path = remove_from_end(imported_full_path, os.sep + INIT_FILE)
        path_parts = imported_full_path.split(os.sep)[-len(module_parts):]
        path_parts[-1] = remove_from_end(path_parts[-1], PY_EXTENSION)
        if path_parts == module_parts:
            return False
        else:
            return True

    def get_destination(self, full_path_relative, dir_module, relative_module):
        full_path = os.path.abspath(full_path_relative)
        module_name = ""
        parent_dir = self.get_module_root_dir(full_path)

        pack_name = ntpath.basename(parent_dir)
        path_dest = os.path.abspath(full_path)
        path_dest = path_dest.replace(parent_dir, '')
        if path_dest.startswith(os.sep):
            path_dest = path_dest[1:]

        if self.is_single_file_module(full_path):
            module_name = os.path.basename(full_path)
        elif not module_name:
            module_name = os.path.basename(os.path.abspath(dir_module))
        if parent_dir == dir_module:
            destination = path_dest
            module_name = pack_name
        else:
            if path_dest:
                destination = path_dest + " (" + parent_dir + ")"
                module_name = pack_name
            else:
                destination = pack_name + " (" + parent_dir + ")"

        if self.is_single_file_module(full_path):
            return destination
        return os.path.join(module_name, destination)

    def get_module_root_dir(self, full_path):
        valid_paths = [module_path for module_path in self.site_packages.values() if
                       os.path.abspath(module_path) in full_path]
        if len(valid_paths) == 0:
            return full_path
        valid_path_max = valid_paths[0]
        for valid_path in valid_paths:
            if len(valid_path) > len(valid_path_max):
                valid_path_max = valid_path
        return valid_path_max

    def get_full_path(self, module, dir_module, origin_file_path, import_level, level_and_names):
        out_module = ''
        imported_full_paths_to_names = {}
        external_module = module.split('.')[0]
        mdl_as_path_file = get_relative_path(import_level) + module.replace('.', os.sep)

        internal_file_path_folder = os.path.normpath(os.path.join(dir_module, mdl_as_path_file))
        relative_file_path_folder = os.path.normpath(
            os.path.join(os.path.abspath(os.path.dirname(origin_file_path)), mdl_as_path_file))

        internal_file_path_file = internal_file_path_folder + PY_EXTENSION
        external_module_key_in_site_packages = external_module.lower()
        if external_module_key_in_site_packages in self.site_packages and import_level == 0:
            closest_folder = self.site_packages[external_module_key_in_site_packages]
            site_package_folder = os.path.dirname(closest_folder)
            external_file_path = os.path.join(site_package_folder, module.replace('.', os.sep) + PY_EXTENSION)
            relative_inner_folder = os.path.join(site_package_folder, module.replace(".", os.sep))
            if os.path.isfile(external_file_path):
                if not external_file_path.startswith(dir_module):
                    ex_mdl = os.path.join(site_package_folder, external_module)
                    if ex_mdl not in self.visited:
                        out_module = ex_mdl
                if level_and_names and level_and_names[0][1] == '*':
                    self.add_name_to_path(imported_full_paths_to_names, to_init_file_if_folder(external_file_path), '*')
                else:
                    self.add_name_to_path(imported_full_paths_to_names, to_init_file_if_folder(external_file_path), "")
            elif os.path.exists(relative_inner_folder):
                if relative_inner_folder not in self.visited:
                    out_module = self.site_packages[module.split(".")[0].lower()]
                init_file = (to_init_file_if_folder(relative_inner_folder))
                if level_and_names:
                    self.add_names_mappings(imported_full_paths_to_names, relative_inner_folder, level_and_names)
                else:
                    self.add_name_to_path(imported_full_paths_to_names, init_file, "")
            elif os.path.exists(
                    os.path.join(closest_folder, external_module)):
                relative_inner_folder = os.path.join(closest_folder, external_module)

                if os.path.isfile(internal_file_path_file):
                    self.add_name_to_path(imported_full_paths_to_names, internal_file_path_file, "")

                if level_and_names:
                    self.add_names_mappings(imported_full_paths_to_names, relative_inner_folder, level_and_names)

                if closest_folder not in self.visited:
                    if os.path.exists(closest_folder):
                        out_module = closest_folder
                        self.add_name_to_path(imported_full_paths_to_names,
                                              to_init_file_if_folder(relative_inner_folder), "")
        else:
            path = relative_file_path_folder if import_level > 0 else internal_file_path_folder
            handle_path_result = self.handle_module_by_path(imported_full_paths_to_names, level_and_names, path,
                                                            import_level > 0)
            if not handle_path_result:
                path, parent_module = self.get_from_site_packages(external_module)
                if path is not None:
                    if os.path.isdir(path):
                        self.add_name_to_path(imported_full_paths_to_names, to_init_file_if_folder(path), "")
                    self.handle_module_by_path(imported_full_paths_to_names, level_and_names, path, import_level > 0)
                    out_module = parent_module
        return imported_full_paths_to_names, out_module

    def add_name_to_path(self, imported_full_paths_to_names, path, name):
        if path not in imported_full_paths_to_names:
            imported_full_paths_to_names[path] = []
        imported_full_paths_to_names[path].append(name)

    def handle_module_by_path(self, imported_full_paths, names, path, is_relative_import):
        path_with_py_ext = path + PY_EXTENSION
        if names and (os.path.exists(path) or os.path.exists(path_with_py_ext)):
            self.add_names_mappings(imported_full_paths, path, names)
            return True
        if os.path.isdir(path) and os.path.isfile(os.path.join(path, INIT_FILE)):
            init_file_path = os.path.join(path, INIT_FILE)
            self.add_name_to_path(imported_full_paths, init_file_path, "")
            return True
        if os.path.isfile(path_with_py_ext):
            self.add_name_to_path(imported_full_paths, path_with_py_ext, "")
            return True
        nearby_init_file = os.path.join(os.path.dirname(path), INIT_FILE)
        base_name = os.path.basename(path)
        if os.path.isfile(nearby_init_file) and is_relative_import and \
                (base_name in self.get_objects_in_package_init(nearby_init_file)):
            self.add_name_to_path(imported_full_paths, nearby_init_file, "")
        return False

    def add_names_mappings(self, imported_full_paths_to_names, from_path, level_and_names):
        for level_and_name in level_and_names:
            name = level_and_name[1]
            combined_as_dir = os.path.join(from_path, name)
            init_of_combined_as_dir = to_init_file_if_folder(combined_as_dir)
            combined_as_file = os.path.join(from_path, name + PY_EXTENSION)
            from_as_module = from_path + PY_EXTENSION
            from_as_package = os.path.join(from_path, INIT_FILE)
            if os.path.isdir(combined_as_dir) and is_file_case_sensitive(init_of_combined_as_dir):
                self.add_name_to_path(imported_full_paths_to_names, init_of_combined_as_dir, name)
            elif is_file_case_sensitive(combined_as_file):
                self.add_name_to_path(imported_full_paths_to_names, combined_as_file, name)
            elif is_file_case_sensitive(from_as_module):
                self.add_name_to_path(imported_full_paths_to_names, from_as_module, name)
            elif is_file_case_sensitive(from_as_package):
                self.add_name_to_path(imported_full_paths_to_names, from_as_package, name)

    def get_objects_in_package_init(self, init_file):
        if not (init_file in self.init_file_to_objects):
            self.init_file_to_objects[init_file] = get_objects(init_file)
        return self.init_file_to_objects[init_file]

    # for future use (ANA-1007) instead of FileParser.get_objects
    # importlib can be used when the libraries are installed in the via environment
    #
    # import importlib.util
    # def get_objects_in_package_init(self, init_file):
    #   if is_file_case_sensitive(os.path.dirname(init_file)):
    #       spec = importlib.util.spec_from_file_location(os.path.dirname(init_file), init_file)
    #       module = importlib.util.module_from_spec(spec)
    #       spec.loader.exec_module(module)
    #       if name in dir(module):
    #           self.add_name_to_path(os.path.dirname(init_file), init_file, name)

    def get_from_site_packages(self, external_module):
        for site in self.site_packages:
            site_path = self.site_packages[site]
            new_site_path = site_path + os.sep + external_module
            if os.path.exists(new_site_path):
                return new_site_path, site_path
            new_site_path = site_path + os.sep + external_module + PY_EXTENSION
            if os.path.exists(new_site_path):
                return new_site_path, site_path
        return None, None

    def is_single_file_module(self, full_path):
        return full_path in self.site_packages.values()


def is_file_case_sensitive(path):
    dirname, filename = os.path.split(path)
    if not os.path.isfile(path):
        return False
    if not os.path.isdir(dirname):
        return False
    return filename in os.listdir(dirname)
