import groovy.json.JsonOutput
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ModuleVersionIdentifier
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.artifacts.dsl.DependencyHandler
import org.gradle.api.artifacts.result.ResolvedComponentResult
import org.gradle.api.artifacts.result.ResolvedDependencyResult
import org.gradle.api.artifacts.result.UnresolvedDependencyResult
import org.gradle.maven.MavenModule
import org.gradle.maven.MavenPomArtifact

def executed = false
gradle.allprojects {
    task whitesourceDependenciesTask { task ->
        task.doLast {
            if (executed) {
                // we configured this task for all projects,
                // but we only want to run it once
                return
            }

            executed = true

            println("WHITESOURCE: dependencies  task - START")

            List<GradleProject> projects = []
            rootProject.allprojects { Project p ->
                def gp = new GradleProject()
                gp.groupId = p.group
                gp.artifactId = p.name
                if (p.version != "unspecified") gp.version = p.version
                gp.manifest = p.buildFile.getAbsolutePath()

                if (System.getProperty("WS_INIT_GRADLE_INCLUDE_DEPENDENCIES", "false").toBoolean()) {
                    gp.configGraphs = p.configurations.findAll { it.canBeResolved }.collectEntries { config ->
                        def graph = new ConfigGraph(configuration: config, dependencyHandler: p.dependencies)
                        graph.extract(config.incoming.resolutionResult.root)
                        graph?.directDeps ? [(config.name): graph] : [:]
                    }
                }
                projects.add(gp)
            }

            // This map is used in buildDeps for determining inner module dependencies
            Map<String, String> projectPathToGAV = rootProject.allprojects.collectEntries { Project p ->
                [("project " + p.path): "$p.group:$p.name:$p.version".toString()]
            }

            buildDeps(projects, projectPathToGAV)

            println("WHITESOURCE: dependencies task - END")

            printJson(projects)
        }
    }
}

static def printJson(List<GradleProject> projects) {
    def outputDir = new File(System.getProperty("WS_INIT_GRADLE_OUTPUT_PATH", "./dependencies"))
    outputDir.mkdirs()
    println("WHITESOURCE: printing json files to output dir: " + outputDir)
    for (i in 0..<projects.size()) {
        def project = projects.get(i)
        def projectJson = JsonOutput.toJson(project)
        def jsonFile = new File(outputDir, i + ".json")
        jsonFile.write(projectJson)
    }
}

static def buildDeps(List<GradleProject> projects, Map<String, String> projectDisplayNameToGAV) {
    println("WHITESOURCE: buildDeps - START")
    projects.each { GradleProject gp ->
        gp.configGraphs.values().each { ConfigGraph graph ->
            Set<ResolvedArtifact> artifacts
            try {
                artifacts = graph.theConfiguration().resolvedConfiguration.lenientConfiguration.getArtifacts()
            } catch (ignored) {
                println(ignored)
                //return
            }

            graph.deps2Children.keySet().each { String depName ->
                if (!gp.dependencies[depName]?.systemPath && !projectDisplayNameToGAV.containsKey(depName)) {
                    def dep = buildDep(depName, artifacts as Set<ResolvedArtifact>, graph.theDependencyHandler(), projectDisplayNameToGAV)
                    gp.dependencies.put(depName, dep)
                }
            }
        }

        // Here we fix up cases where a dependency uses the project display name.
        // In this case we have already created a dependency object with the project gav in the above code,
        // so we point the display name to that dependency in order to avoid duplicate dependency objects
        projectDisplayNameToGAV.each {
            def displayName = it.key
            def gav = it.value
            def gavDep = gp.dependencies.get(gav)
            if (gavDep) {
                gp.dependencies.put(displayName, gavDep)
            }
        }

        println("WHITESOURCE: buildDeps - $gp - ${gp.dependencies.keySet().size()}")
    }
    println("WHITESOURCE: buildDeps - END")
}

static def buildDep(
        String name,
        Set<ResolvedArtifact> artifacts,
        DependencyHandler handler,
        Map<String, String> projectPathToGAV
) {
    def (g, a, v) = extractGAV(name)

    def d = new Dependency(groupId: g, artifactId: a, version: v != "unspecified" ? v : "")

    // we check if the dependency name matches a project GAV
    // if it does, we know it's an inner module
    if (projectPathToGAV.values().find { it == name }) d.isInnerModule = true

    def artifact = artifacts.find {
        def id = it.moduleVersion.id
        [id.group, id.name, id.version] == [g, a, v] && it.file.isFile()
    }

    if (artifact) {
        d.systemPath = artifact.file.absolutePath
        d.type = artifact.type
    } else if (!d.isInnerModule) {
        // if there is no artifact for this dependency, but it has been successfully resolved,
        // it must be a pom.xml dependency. So we build a custom query for the pom.xml path
        def queryResult = handler.createArtifactResolutionQuery()
                .forModule(d.groupId, d.artifactId, d.version)
                .withArtifacts(MavenModule, MavenPomArtifact)
                .execute()

        if (queryResult.resolvedComponents) {
            def component = queryResult.resolvedComponents[0]
            if (component && component.getArtifacts(MavenPomArtifact)[0].hasProperty('file')) {
                def pomFile = component.getArtifacts(MavenPomArtifact)[0].file
                d.type = "pom"
                d.systemPath = pomFile.getAbsolutePath()
            }
        }
    }

    return d
}

static List<String> extractGAV(String name) {
    ((name.split(":") + ["", "", ""]) as List).subList(0, 3)
}

// Represents a gradle project
class GradleProject {
    String groupId
    String artifactId
    String version
    String manifest

    Map<String, Dependency> dependencies = [:]
    Map<String, ConfigGraph> configGraphs = [:]

    String toString() {
        def suffix = version ? ":$version" : ""
        return "$groupId:$artifactId$suffix".toString()
    }
}

// Represents a minimal DependencyInfo object, with some extra metadata
class Dependency {
    String groupId
    String artifactId
    String version
    String systemPath
    String type

    boolean isInnerModule
}

// Contains the data needed to build the dependency graph for a configuration
class ConfigGraph {

    Set<String> directDeps = []
    Map<String, Set<String>> deps2Children = [:]

    // reference to the Configuration
    // we use it to find the artifact locations
    private Configuration configuration

    // unconventional getter name so that JsonOutput will ignore it
    def theConfiguration() { configuration }

    // reference to the gradle project dependency handler
    // we use it to find the location of pom artifacts if a normal artifact is not found
    private DependencyHandler dependencyHandler

    // unconventional getter name so that JsonOutput will ignore it
    def theDependencyHandler() { dependencyHandler }


    def extract(ResolvedComponentResult root) {
        root.dependencies.each { extract(it, true) }
    }

    String extract(result, direct = false) {
        String name
        if (result instanceof UnresolvedDependencyResult) {
            name = extractUnresolved((UnresolvedDependencyResult) result)
        } else {
            name = extractResolved((ResolvedDependencyResult) result)
        }

        // Only add as direct dependency if it's NOT a constraint
        if (direct && !result.isConstraint()) {
            directDeps.add(name)
        }

        return name
    }

    String extractUnresolved(UnresolvedDependencyResult result) {
        def name = result.attempted.displayName
        if (deps2Children.containsKey(name)) return name
        deps2Children[name] = new HashSet()
        return name
    }

    String extractResolved(ResolvedDependencyResult result) {
        def name = depName(result.selected.moduleVersion)
        if (deps2Children.containsKey(name)) return name
        deps2Children[name] = new HashSet()
        result.selected.dependencies.each {
            def child = extract(it)
            deps2Children[name].add(child)
        }
        return name
    }

    static String depName(ModuleVersionIdentifier m) {
        return "$m.group:$m.name:$m.version".toString()
    }
}
