package com.wss.scanner.registry.utils.registryHandlers.gradle;

import com.wss.common.logging.LogContext;
import com.wss.scanner.registry.models.HostRule;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.LinkedList;
import java.util.List;

import static com.wss.scanner.registry.utils.Constants.*;
import static com.wss.scanner.registry.utils.Constants.CLOSE_CURLY_BRACKETS;

public class GradleCredentialsBlocksCreator {
    private final static Logger logger = LoggerFactory.getLogger(GradleCredentialsBlocksCreator.class);
    private final static String AUTHORIZATION = "Authorization";
    private final static String HTTP_HEADER_CREDENTIALS = "HttpHeaderCredentials";
    private final static String HTTP_HEADER_AUTHENTICATION = "HttpHeaderAuthentication";
    private final static String BASIC_AUTHENTICATION = "BasicAuthentication";
    private final static String CREATE = "create";
    private final static String BASIC = "basic";
    private final static String HEADER = "header";
    private final static String BEARER = "Bearer";
    private final static String UNESCAPED_DOLLAR = "\\$";
    private final static String ESCAPED_DOLLAR = "\\\\\\$";

    /**
     * In this method we build the full dependencies/plugin management block
     * @param logContext    - the log context
     * @param hostRules     - the host rules
     * @param isKotlin      - whether the registry file is Kotlin
     * @param isContainsIvy - whether the repositories contain Ivy
     * @param blockKey      - the block key, like "allprojects" or "dependencyResolutionManagement"
     * @return the dependencies/plugin management block
     */
    public List<String> prepareCredentialsManagementBlock(LogContext logContext, List<HostRule> hostRules,
                                                           boolean isKotlin, boolean isContainsIvy, String blockKey) {
        List<String> lines = new LinkedList<>();
        List<String> repositoriesBlock = prepareRepositoriesBlock(logContext, hostRules, isKotlin, isContainsIvy);
        lines.add(blockKey + SPACE + OPEN_CURLY_BRACKETS);
        lines.addAll(repositoriesBlock);
        lines.add(CLOSE_CURLY_BRACKETS);
        return lines;
    }

    /**
     * In this method we build the whole repositories block example:
     * repositories {
     *     		maven {
     * 			  url "url"
     * 			  credentials {
     * 				username "username"
     * 				password "password"
     *            }
     *          }
     * }
     * @param logContext    - the log context
     * @param hostRules     - the host rules
     * @param isKotlin      - whether the registry file is Kotlin
     * @param isContainsIvy - whether the repositories contain Ivy
     * @return the repositories block with the host rules credentials
     */
    public List<String> prepareRepositoriesBlock(LogContext logContext, List<HostRule> hostRules, boolean isKotlin,
                                                  boolean isContainsIvy) {
        List<String> repositories = new LinkedList<>();

        repositories.add(REPOSITORIES + SPACE + OPEN_CURLY_BRACKETS);
        repositories.add("mavenCentral()");
        repositories.addAll(prepareRegistriesCredintials(logContext, hostRules, isKotlin, isContainsIvy));
        repositories.add(CLOSE_CURLY_BRACKETS);

        return repositories;
    }

    /**
     *In this method we build the registries credentials block, for example:
     * maven {
     *      url "url"
     *      credentials {
     *         username "username"
     *         password "password"
     *      }
     * }
     * @param logContext    - the log context
     * @param hostRules     - the host rules
     * @param isKotlin      - whether the registry file is Kotlin
     * @param isContainsIvy - whether the repositories contain Ivy
     * @return the credentials block
     */
    public List<String> prepareRegistriesCredintials(LogContext logContext, List<HostRule> hostRules, boolean isKotlin,
                                                     boolean isContainsIvy) {
        List<String> registriesLines = new LinkedList<>();

        for (HostRule hostRule : hostRules) {
            registriesLines.add(MAVEN + SPACE + OPEN_CURLY_BRACKETS);
            List<String> credentials = prepareCredentialsBlock(logContext, hostRule, isKotlin);
            registriesLines.addAll(credentials);
            registriesLines.add(CLOSE_CURLY_BRACKETS);

            if (isContainsIvy) {
                registriesLines.add(IVY + SPACE + OPEN_CURLY_BRACKETS);
                registriesLines.addAll(credentials);
                registriesLines.add(CLOSE_CURLY_BRACKETS);
            }
        }

        return registriesLines;
    }

    /* private methods */

    /**
     * In this method we build the credentials block, for example:
     * 	url "url"
     * 	credentials(HttpHeaderCredentials) {
     * 		name "Private-Token"
     * 		value "token"
     *  }
     * 	authentication {
     * 		header(HttpHeaderAuthentication)
     * 	}
     * @param logContext - the log context
     * @param hostRule   - the host rules list
     * @param isKotlin   - whether the registry file is Kotlin
     * @return the credentials block
     */
    private List<String> prepareCredentialsBlock(LogContext logContext, HostRule hostRule, boolean isKotlin) {
        List<String> credentials = new LinkedList<>();
        String url = buildUrlLine(hostRule.getMatchHost(), isKotlin);
        credentials.add(url);

        String credentialsLine = "";
        String name = "";
        String tokenOrPassword = "";
        List<String> authBlock = new LinkedList<>();

        String token = hostRule.getToken();
        if (StringUtils.isNotBlank(token)) {
            credentialsLine = buildCredentialsLine(false, isKotlin);
            name = buildCredentialsValuesLine(NAME, AUTHORIZATION, true);
            tokenOrPassword = buildCredentialsValuesLine(VALUE, BEARER + SPACE + token, true);
            authBlock = buildAuthenticationBlock(logContext, HTTP_HEADER_AUTHENTICATION, HEADER, isKotlin);
        } else if (StringUtils.isNotBlank(hostRule.getUserName()) && StringUtils.isNotBlank(hostRule.getPassword())) {
            credentialsLine = buildCredentialsLine(true, isKotlin);
            name = buildCredentialsValuesLine(USERNAME, hostRule.getUserName(), isKotlin);
            tokenOrPassword = buildCredentialsValuesLine(PASSWORD, hostRule.getPassword(), isKotlin);
            authBlock = buildAuthenticationBlock(logContext, BASIC_AUTHENTICATION, BASIC, isKotlin);
        }

        credentials.add(credentialsLine);
        credentials.add(name);
        credentials.add(tokenOrPassword);
        credentials.add(CLOSE_CURLY_BRACKETS);
        credentials.addAll(authBlock);

        return credentials;
    }

    /**
     * In this method we return the url of the credentials section, for example:
     * for Kotlin: url = uri("http://repo.mycompany.com/maven2")
     * for groovy: url "http://repo.mycompany.com/maven2"
     * @param hostUrl  - the host url
     * @param isKotlin - whether the registry file is Kotlin
     * @return
     */
    private String buildUrlLine(String hostUrl, boolean isKotlin) {
        if (hostUrl.contains(DOLLAR)) {
            hostUrl = escapeEachDollarSign(hostUrl);
        }
        if (isKotlin) {
            return URL + SPACE + EQUALS + SPACE + URI + OPEN_ROUND_BRACKETS + QUOTATION_MARK +
                    hostUrl + QUOTATION_MARK + CLOSE_ROUND_BRACKETS;
        } else {
            return URL + SPACE + QUOTATION_MARK + hostUrl + QUOTATION_MARK;
        }
    }

    /**
     * In this method we return inner credentials line, for example:
     * name = "Private-Token" or username = "user"
     * value = "TOKEN" or password = "password"
     * IMPORTANT NOTE: Gradle might add value with $ which will be considered as environment variable which will cause
     * the build to break and potentially print sensitive information to the logs.
     * In case the value contains '$', we will escape it by adding '\'.
     * @param key      - the credentials name
     * @param value    - the credentials value
     * @param isKotlin - whether the registry file is Kotlin
     * @return the credentials name/value line
     */
    private String buildCredentialsValuesLine(String key, String value, boolean isKotlin) {
        if (value.contains(DOLLAR)) {
            value = escapeEachDollarSign(value);
        }
        if (isKotlin) {
            return key + SPACE + EQUALS + SPACE + QUOTATION_MARK + value + QUOTATION_MARK;
        } else {
            return key + SPACE + QUOTATION_MARK + value + QUOTATION_MARK;
        }
    }

    /**
     * In this method we escape each dollar sign $ by adding a "\" before it
     * @param strWithDollarSign - the string value containing the $
     * @return the same string with each dollar sign escaped
     */
    private String escapeEachDollarSign(String strWithDollarSign) {
        return strWithDollarSign.replaceAll(UNESCAPED_DOLLAR, ESCAPED_DOLLAR);
    }

    /**
     * In this method we build the credentials line of each credential's block
     * examples:
     * "credentials(HttpHeaderCredentials) {" or "credentials {"
     * @param isBasicAuth - whether we're using username/password
     * @param isKotlin    - whether the file is kotlin
     * @return the credentials line
     */
    private String buildCredentialsLine(boolean isBasicAuth, boolean isKotlin) {
        if (isBasicAuth) {
            return CREDENTIALS + SPACE + OPEN_CURLY_BRACKETS;
        } else {
            String common = CREDENTIALS + OPEN_ROUND_BRACKETS + HTTP_HEADER_CREDENTIALS;
            if (isKotlin) {
                return common + TWO_COLONS + CLASS + CLOSE_ROUND_BRACKETS + SPACE + OPEN_CURLY_BRACKETS;
            } else {
                return common + CLOSE_ROUND_BRACKETS + SPACE + OPEN_CURLY_BRACKETS;
            }
        }
    }

    /**
     * In this method we build the authentication block of each of the new credentials blocks
     * @param logContext - the log context
     * @param authClass  - the authentication class
     * @param authType   - the authentication type
     * @param isKotlin   - whether registry file kotlin
     * @return list of authentication block lines
     */
    private List<String> buildAuthenticationBlock(LogContext logContext, String authClass, String authType,
                                                  boolean isKotlin) {

        List<String> authLines = new LinkedList<>();
        authLines.add(AUTHENTICATION + SPACE + OPEN_CURLY_BRACKETS);
        if (isKotlin) {
            authLines.add(CREATE + LESS_THAN + authClass + GREATER_THAN + OPEN_ROUND_BRACKETS + QUOTATION_MARK +
                    authType + QUOTATION_MARK + CLOSE_ROUND_BRACKETS);
        } else {
            authLines.add(authType + OPEN_ROUND_BRACKETS + authClass + CLOSE_ROUND_BRACKETS);
        }
        authLines.add(CLOSE_CURLY_BRACKETS);

        return authLines;
    }
}
