package com.wss.scanner.registry.utils.registryHandlers.gradle;

import com.wss.common.logging.LogContext;
import com.wss.common.logging.LogUtils;
import com.wss.scanner.registry.models.HostRule;
import com.wss.scanner.registry.utils.Constants;
import com.wss.scanner.registry.utils.OsUtils;
import com.wss.scanner.registry.utils.PrivateRegistryUtils;
import com.wss.scanner.registry.utils.filesParsers.GradleFilesParser;
import com.wss.scanner.registry.utils.helpers.Pair;
import com.wss.scanner.registry.utils.registryHandlers.PrivateRegistryFileHandler;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileWriter;
import java.io.PrintWriter;
import java.lang.reflect.Field;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;
import java.io.IOException;

import static com.wss.scanner.registry.utils.Constants.*;

public class GradlePrivateRegistryHandler extends PrivateRegistryFileHandler {
    private final static Logger logger = LoggerFactory.getLogger(GradlePrivateRegistryHandler.class);
    private final static String DOT_KTS = ".kts";

    @Override
    public void createRegistryObject(LogContext logContext, String filePath, List<HostRule> hostRules) {
        // we don't create any build.gradle file, as we always write the new host rules to the existing build.gradle file
    }

    @Override
    public void editRegistryObject(LogContext logContext, String registryFilePath, List<HostRule> hostRules) {
        boolean hasEnvVariablesMapping = hostRules.stream()
                .map(HostRule::getEnvVariablesMapping)
                .filter(Objects::nonNull)
                .findFirst()
                .isPresent();
        if (hasEnvVariablesMapping) {
            setGradleEnvironmentVariables(logContext, hostRules);
        } else {
            boolean isKotlin = registryFilePath.endsWith(DOT_KTS);
            boolean isBuildGradle = registryFilePath.endsWith(BUILD_GRADLE) || registryFilePath.endsWith(BUILD_GRADLE_KTS);
            GradleFilesParser gradleFilesParser = new GradleFilesParser();
            Path path = Paths.get(registryFilePath);
            List<String> registryFileLines = gradleFilesParser.parseGroovyFile(logContext, registryFilePath);

            registryFileLines = cleanRegistryFileFromUndefinedEnvVar(logContext, path.getParent().toString(), registryFileLines);
            editRegistryFile(logContext, hostRules, isKotlin, isBuildGradle, registryFileLines);
            PrivateRegistryUtils.writeLinesToFile(logContext, path, registryFileLines);
        }
    }

    @Override
    public boolean isProjectLevelRegistryFile() {
        return true;
    }

    @Override
    public boolean isSystemLevelRegistryFile() {
        return false;
    }

    @Override
    public boolean isEnvVariableRegistryCredentials() {
        return false;
    }

    @Override
    public void prepareIncludesAndExcludes(ArrayList<String> includes, ArrayList<String> excludes) {
        String[] jsIncludedManifests = new String[]{
                BUILD_GRADLE,
                BUILD_GRADLE_KTS,
                SETTINGS_GRADLE,
                SETTINGS_GRADLE_KTS
        };

        includes.addAll(Arrays.asList(jsIncludedManifests));
    }

    @Override
    public List<String> getManifestTypes(String[] localManifestsFiles) {
        return Arrays.asList(BUILD_GRADLE, BUILD_GRADLE_KTS, SETTINGS_GRADLE, SETTINGS_GRADLE_KTS);
    }

    @Override
    public List<String> getGlobalRegistryObject(boolean isFile) {
        return null;
    }

    @Override
    public List<String> getRegistryFileType(String manifestFile) {
        return Arrays.asList(manifestFile);
    }

    /* private methods */

    /**
     * In this method we add/edit the specific block of lines that holds the repositories block
     *
     * @param logContext - the log context
     * @param blockLines - the specific block of lines we want to edit
     * @param hostRules  - the host rules
     * @param isKotlin   - if the registry file is kotlin
     * @return the modified block list of lines
     */
    private List<String> editGradleRepositoriesBlock(LogContext logContext, List<String> blockLines,
                                                     List<HostRule> hostRules, boolean isKotlin) {
        List<String> editedBlockLines = new LinkedList<>(blockLines);
        try {
            boolean isRepositoriesExist = false;
            boolean isContainsIvy = false;
            int repositoriesKeyInd = -1;

            for (int i = 0; i < blockLines.size(); i++) {
                String line = blockLines.get(i);
                if (line.trim().contains(REPOSITORIES)) {
                    isRepositoriesExist = true;
                    repositoriesKeyInd = i;
                } else if (line.contains(IVY)) {
                    isContainsIvy = true;
                }
            }

            GradleCredentialsBlocksCreator gradleCredentialsBlocksCreator = new GradleCredentialsBlocksCreator();
            // the repositories block is missing from the block lines
            if (!isRepositoriesExist) {
                List<String> repositoriesBlock = gradleCredentialsBlocksCreator.prepareRepositoriesBlock(logContext,
                        hostRules, isKotlin, isContainsIvy);
                // add the new repositories block after the parent key, for example, add it after "allproject {"
                editedBlockLines.addAll(1, repositoriesBlock);
                repositoriesKeyInd = 1;
            } else {

                Map<String, Pair<Integer, Integer>> blocksIndexes = findBlocksIndexes(Collections.singletonList(REPOSITORIES), editedBlockLines);
                Integer repositoriesLastIndex = blocksIndexes.get(REPOSITORIES).getValue();
                // prepare the new repositories credentials (not the whole block)
                List<String> hostRulesBlocks = gradleCredentialsBlocksCreator.prepareRegistriesCredintials(logContext,
                        hostRules, isKotlin, isContainsIvy);
                // add the new credentials after the original repository block, since gradle uses the latest credentials "
                editedBlockLines.addAll(repositoriesLastIndex, hostRulesBlocks);
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - editGradleRepositoriesBlock - error while editing repositories block "));
        }
        return editedBlockLines;
    }

    /**
     * In this method we edit the registry file two blocks (if found), the "allprojects" and "buildscript" blocks
     * if any of these blocks are not found, we create them from scratch and push the "buildscript" to the front of the
     * registry lines list and the "allprojects" to the end of the list
     * if any these blocks are found, we edit them and replace the existing once by the new ones
     * please note that the "buildscript" must be at the beginning of the registry file, and the "allprojects" must not
     * be in front of the "plugins" block (if found)
     *
     * @param logContext        - the log context
     * @param hostRules         - the host rules
     * @param isKotlin          - whether the registry file is kotlin or not
     * @param registryFileLines - registry file lines
     */
    private void editRegistryFile(LogContext logContext, List<HostRule> hostRules, boolean isKotlin, boolean isBuildGradle,
                                  List<String> registryFileLines) {
        try {
            List<String> registriesBlocks;
            if (isBuildGradle) {
                registriesBlocks = Arrays.asList(ALL_PROJECTS, BUILD_SCRIPT);
            } else {
                registriesBlocks = Arrays.asList(DEPENDENCY_RESOLUTION_MANAGEMENT, PLUGIN_MANAGEMENT);
            }

            Map<String, Pair<Integer, Integer>> blocksIndexes = findBlocksIndexes(registriesBlocks, registryFileLines);

            GradleCredentialsBlocksCreator gradleCredentialsBlocksCreator = new GradleCredentialsBlocksCreator();
            if (blocksIndexes.isEmpty()) {
                List<String> dependenciesCredentialsBlock = gradleCredentialsBlocksCreator.prepareCredentialsManagementBlock(
                        logContext, hostRules, isKotlin, false, registriesBlocks.get(0));
                List<String> pluginCredentialsBlock = gradleCredentialsBlocksCreator.prepareCredentialsManagementBlock(
                        logContext, hostRules, isKotlin, false, registriesBlocks.get(1));

                registryFileLines.addAll(0, pluginCredentialsBlock); // must be at the beginning of the registry file
                registryFileLines.addAll(dependenciesCredentialsBlock);
                return;
            }

            Map<String, List<String>> registriesBlockLine = new HashMap<>();
            for (String registriesBlock : registriesBlocks) {
                if (blocksIndexes.containsKey(registriesBlock)) {
                    Pair<Integer, Integer> blockIndexes = blocksIndexes.get(registriesBlock);
                    List<String> modifiedBlockLines = editGradleRepositoriesBlock(logContext,
                            registryFileLines.subList(blockIndexes.getKey(), blockIndexes.getValue() + 1), hostRules, isKotlin);
                    registriesBlockLine.put(registriesBlock, modifiedBlockLines);
                } else {
                    List<String> missingBlockLines = gradleCredentialsBlocksCreator.prepareCredentialsManagementBlock(
                            logContext, hostRules, isKotlin, false, registriesBlock);
                    registriesBlockLine.put(registriesBlock, missingBlockLines);
                }
            }

            /* if the allprojects is found, then we need to remove the existing block and replace it
             with the block that we created/edit
             the order of the for is important, as we want to remove the later object so the indexes of the other
             block won't change
             */

            //Sort descending by start line  so clearing lines won't affect indices.
            List<Pair<Integer, Integer>> sorted = blocksIndexes.values().stream()
                    .sorted((o1, o2) -> -1 * o1.getKey().compareTo(o2.getKey())).collect(Collectors.toList());
            for (Pair<Integer, Integer> blockIndex : sorted) {
                int startId = blockIndex.getKey();
                int endId = blockIndex.getValue();
                registryFileLines.subList(startId, endId + 1).clear();
            }

            if (isBuildGradle || registriesBlocks.size() > 1) {
                // allprojects or dependencyResolutionManagement
                registryFileLines.addAll(registriesBlockLine.get(registriesBlocks.get(0)));

                // buildscript or pluginManagement
                registryFileLines.addAll(0, registriesBlockLine.get(registriesBlocks.get(1)));
            } else {
                registryFileLines.addAll(0, registriesBlockLine.get(registriesBlocks.get(1)));
            }
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - editRegistryFile - error while editing registry file lines "));
        }
    }

    /**
     * In this method we build the gradle properties and env variable map and call the removeUndefinedEnvVariables method
     *
     * @param logContext        - the log context
     * @param projectPath       - the project path
     * @param registryFileLines - the registry file parsed lines
     */
    private List<String> cleanRegistryFileFromUndefinedEnvVar(LogContext logContext, String projectPath, List<String> registryFileLines) {
        try {
            GradleConfigurationsUtils gradleConfigurationsUtils = new GradleConfigurationsUtils();
            Map<String, String> gradleProperties = gradleConfigurationsUtils.buildGradleProperties(logContext,
                    projectPath, registryFileLines);
            return removeUndefinedEnvVariables(logContext, registryFileLines, gradleProperties);
        } catch (Exception e) {
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, this.getClass().getName() +
                    " - cleanRegistryFileFromUndefinedEnvVar - error while collecting properties and removing env variables "));
        }
        return registryFileLines;
    }

    /**
     * In this method we locate and remove any prop variable in the registry lines that is not configured
     *
     * @param logContext    - the log context
     * @param lines         - the lines we want to locate and remove the prop variables from
     * @param propVariables - the env variables
     */
    private List<String> removeUndefinedEnvVariables(LogContext logContext, List<String> lines, Map<String, String> propVariables) {
        PrivateRegistryUtils privateRegistryUtils = new PrivateRegistryUtils();
        List<String> linesAfterRemove = new ArrayList<>();
        for (int i = 0; i < lines.size(); i++) {
            String line = lines.get(i);
            boolean envVariablesExist = privateRegistryUtils.isEnvVariablesExist(logContext, line, propVariables);
            if (!envVariablesExist) {
                logger.warn(LogUtils.formatLogMessage(logContext, "variable in line {} is not" +
                        " defined in any of the properties files or environment variables"), line);
                if (StringUtils.countMatches(line, OPEN_CURLY_BRACKETS) > StringUtils.countMatches(line, CLOSE_CURLY_BRACKETS)) {
                    Stack<Character> stack = new Stack<>();
                    do {
                        line = lines.get(i);
                        for (char c : line.toCharArray()) {
                            if (c == '{') stack.push('{');
                            else if (c == '}') stack.pop();
                        }
                    } while (!stack.isEmpty() && i++ < lines.size());
                }
            } else {
                linesAfterRemove.add(line);
            }
        }

        return linesAfterRemove;
    }

    /**
     * In this method we find the start and end line index of the specified blocks (allprojects and buildscript)
     *
     * @param blocksNames - the list of the blocks name we want to search for
     * @param lines       - the lines we're looking in
     * @return a map of each block name and the corresponding pair of start/end indexes
     */
    private Map<String, Pair<Integer, Integer>> findBlocksIndexes(List<String> blocksNames, List<String> lines) {
        Map<String, Pair<Integer, Integer>> blockIndexes = new HashMap<>();
        int runningInd = 0;
        int bracketsCounter = 0;
        int startInd = 0;
        int endInd = 0;
        boolean insideGivenBlock = false;
        String key = null;
        while (runningInd < lines.size()) {
            String currentLine = lines.get(runningInd);
            if (currentLine.contains(OPEN_CURLY_BRACKETS)) {
                String blockKey = currentLine.trim().split("\\s+")[0];
                if (blocksNames.contains(blockKey)) {
                    key = blockKey;
                    insideGivenBlock = true;
                    bracketsCounter++;
                    startInd = runningInd;
                } else if (insideGivenBlock) {
                    bracketsCounter++;
                }
            } else if (currentLine.contains(CLOSE_CURLY_BRACKETS) && insideGivenBlock) {
                bracketsCounter--;
            }

            if (bracketsCounter == 0 && insideGivenBlock) {
                insideGivenBlock = false;
                endInd = runningInd;
                blockIndexes.put(key, new Pair<>(startInd, endInd));
            }
            runningInd++;
        }
        return blockIndexes;
    }

    private void setGradleEnvironmentVariables(LogContext logContext, List<HostRule> hostRules) {
        try {
            for (HostRule hostRule : hostRules) {
                Map<String, String> envVariablesMapping = hostRule.getEnvVariablesMapping();
                if (envVariablesMapping != null) {
                    for (Map.Entry<String, String> envVariable : envVariablesMapping.entrySet()) {
                        String variableValue = null;
                        String variableKey = envVariable.getKey();
                        switch (variableKey) {
                            case USERNAME:
                            case "userName":
                                variableValue = hostRule.getUserName();
                                break;
                            case PASSWORD:
                                variableValue = hostRule.getPassword();
                                break;
                            case TOKEN:
                                variableValue = hostRule.getToken();
                                break;
                        }
                        // If the variable value is not null, set the environment variable
                        if (variableValue != null) {
                            String envVariableName = envVariable.getValue();
                            logger.info("Adding the environment variable " + envVariableName);
                            setEnvironmentVariable(envVariableName, variableValue);
                        }
                    }
                }
            }
        } catch (ReflectiveOperationException e) {
            // Handle any exceptions that may occur during the process
            logger.error(LogUtils.getExceptionErrorMessage(logContext, e, PrivateRegistryUtils.class.getName() +
                    "Issue when trying to set Gradle environment variables of private repo"));
        }
    }

    private void setEnvironmentVariable(String key, String value) throws ReflectiveOperationException {
        Map<String, String> env = System.getenv();
        Field field = env.getClass().getDeclaredField("m");
        field.setAccessible(true);
        ((Map<String, String>) field.get(env)).put(key, value);
    }
}