package com.wss.common.logging;

import java.util.Map;
import java.util.stream.Collectors;

public class LogUtils {

    /* --- Static members --- */

    private static final String LOG_MESSAGE_FORMAT = "\t%s";
    private static final String LOG_MESSAGE_WITH_CONTEXT_FORMAT = "[%s]" + LOG_MESSAGE_FORMAT;
    private static final String SCANNER = "scanner";
    public static final String EQUALS = "=";


    /* --- Public methods --- */

    public static String formatLogMessage(LogContext logCtx, String format) {
        if (logCtx == null) {
            return String.format(LOG_MESSAGE_FORMAT, format);
        } else {
            return String.format(LOG_MESSAGE_WITH_CONTEXT_FORMAT, logCtx, format);
        }
    }

    /**
     * this method extracts an exception's error message (when available; when not - extracts its class name)
     * adds a custom message and a relevant stack-trace (one from a wss class).
     * when requested - it prints the message to the log
     *
     * @param exception     - the exception to handle
     * @param customMessage - a custom message to add before the exception's own message
     * @param logContext    - the log context
     * @return - the exception's error message, combined with the custom-message and
     * a relevant stack-trace information
     */
    public static String getExceptionErrorMessage(LogContext logContext, Exception exception, String customMessage) {
        String message = getExceptionCustomMessage(exception, customMessage);
        return formatLogMessage(logContext, message);
    }

    public static String printCustomLog(LogContext logContext, String key, String value) {
        String customLog = key + EQUALS + value;
        return formatLogMessage(logContext, customLog, "");
    }

    public static String printCustomLog(LogContext logContext, String key, String value, String message) {
        String customLog = key + EQUALS + value;
        return formatLogMessage(logContext, customLog, message);
    }

    public static String printCustomLog(LogContext logContext, Map<String, String> custom) {
        String customLog = custom.entrySet().stream().map(e -> e.getKey() + EQUALS + e.getValue()).collect(Collectors.joining(";"));
        return formatLogMessage(logContext, customLog, "");
    }

    public static String printCustomLog(LogContext logContext, Map<String, String> custom, String message) {
        String customLog = custom.entrySet().stream().map(e -> e.getKey() + EQUALS + e.getValue()).collect(Collectors.joining(";"));
        return formatLogMessage(logContext, customLog, message);
    }

    private static String formatLogMessage(LogContext logContext, String customLog, String message) {
        if (logContext == null) {
            return String.format(LOG_MESSAGE_FORMAT, message);
        } else {
            return String.format(LOG_MESSAGE_WITH_CONTEXT_FORMAT,  logContext.toString().concat(";").concat(customLog), message);
        }
    }

    /* --- Private methods --- */

    /**
     * This method creates a message from the exception message and stack trace
     * @param exception     - the accrued exception
     * @param customMessage - our own custom message
     * @return new message consist from the custom message and the stack trace of the exception
     */
    private static String getExceptionCustomMessage(Exception exception, String customMessage) {
        String error = exception.getMessage() == null ? exception.getClass().getName() : exception.getMessage();
        String methodName = new Throwable().getStackTrace()[1].getMethodName();
        return customMessage + error + " StackTrace : " + getRelevantStackTrace(exception.getStackTrace(), methodName);
    }

    private static String getRelevantStackTrace(StackTraceElement[] stackTrace, String methodName) {
        String output = "";
        for (StackTraceElement stackTraceElement : stackTrace) {
            if (stackTraceElement.getClassName().contains(SCANNER)) {
                output = output.concat(stackTraceElement.getClassName()).concat(":").concat(String.valueOf(stackTraceElement.getLineNumber())).concat("-->");
                if (stackTraceElement.getMethodName().equals(methodName))
                    break;
            }
        }
        int endIndex = output.lastIndexOf("-->");
        if (endIndex != -1) {
            output = output.substring(0, endIndex);
            output = " (".concat(output).concat(")");
        }
        return output;
    }
}
